﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ncm/ncm_ContentStorage.h>

namespace nn { namespace ncm {

    class ContentStorage;
    class ContentMetaDatabase;

    /**
    * @brief ncm サービスに必要なリソースを初期化します。
    */
    void Initialize() NN_NOEXCEPT;

    /**
    * @brief ncm サービスに必要なリソースを開放します。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief ストレージ名の最大長です。
    */
    static const size_t MaxStorageNameLength = 15;

    /**
    * @brief    名前を指定して ContentStorage を作成します。
    *
    * @param[in]    storageId   ContentStorage を作成するストレージ
    *
    * @return   処理の結果が返ります。
    * @retval   ResultUnknownStorage    未知のストレージです。
    *
    * @pre
    *           - Initialize が呼び出されている
    * @post
    *           - 対象ストレージに対して OpenContentStorage が成功する
    */
    Result CreateContentStorage(StorageId storageId) NN_NOEXCEPT;

    /**
    * @brief    名前を指定して ContentMetaDatabase を作成します。
    *
    * @param[in]    storageId   ContentMetaDatabase を作成するストレージ
    *
    * @return   処理の結果が返ります。
    * @retval   ResultUnknownStorage    未知のストレージです。
    *
    * @pre
    *           - Initialize が呼び出されている
    * @post
    *           - 対象ストレージに対して OpenContentMetaDatabase が成功する
    */
    Result CreateContentMetaDatabase(StorageId storageId) NN_NOEXCEPT;

    Result VerifyContentStorage(StorageId storageId) NN_NOEXCEPT;

    Result VerifyContentMetaDatabase(StorageId storageId) NN_NOEXCEPT;

    /**
    * @brief    名前を指定して ContentStorage を取得します。
    *
    * @param[out]   outValue        取得した ContentStorage を書き込むバッファ
    * @param[in]    storageId       取得する ContentStorage が存在するストレージ
    *
    * @return   処理の結果が返ります。
    * @retval   ResultUnknownStorage    未知のストレージです。
    *
    * @pre
    *           - Initialize が呼び出されている
    * @post
    *           - outValue に利用可能な ContentStorage が書き込まれている
    */
    Result OpenContentStorage(ContentStorage* outValue, StorageId storageId) NN_NOEXCEPT;

    /**
    * @brief    名前を指定して ContentMetaDatabase を取得します。
    *
    * @param[out]   outValue        取得した ContentMetaDatabase を書き込むバッファ
    * @param[in]    storageId       取得する ContentMetaDatabase が存在するストレージ
    *
    * @return   処理の結果が返ります。
    * @retval   ResultUnknownStorage    未知のストレージです。
    *
    * @pre
    *           - Initialize が呼び出されている
    * @post
    *           - outValue に利用可能な ContentMetaDatabase が書き込まれている
    */
    Result OpenContentMetaDatabase(ContentMetaDatabase* outValue, StorageId storageId) NN_NOEXCEPT;

    Result CleanupContentMetaDatabase(StorageId storageId) NN_NOEXCEPT;

    Result ActivateContentStorage(StorageId storageId) NN_NOEXCEPT;

    Result InactivateContentStorage(StorageId storageId) NN_NOEXCEPT;

    Result ActivateContentMetaDatabase(StorageId storageId) NN_NOEXCEPT;

    Result InactivateContentMetaDatabase(StorageId storageId) NN_NOEXCEPT;

}}
