﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/fs/fs_File.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_InstallProgress.h>
#include <nn/ncm/ncm_SystemUpdateTaskApplyInfo.h>

namespace nn { namespace ncm {

    struct InstallContentMeta
    {
        std::unique_ptr<char[]> data;
        size_t size;

        InstallContentMetaReader GetReader() const NN_NOEXCEPT
        {
            return InstallContentMetaReader(data.get(), size);
        }

        InstallContentMetaWriter GetWriter() NN_NOEXCEPT
        {
            return InstallContentMetaWriter(data.get(), size);
        }
    };

    class InstallTaskDataBase
    {
    public:

        Result Get(InstallContentMeta* outValue, int index) NN_NOEXCEPT;
        Result Update(const InstallContentMeta& data, int index) NN_NOEXCEPT;
        Result Has(bool* outValue, Bit64 id) NN_NOEXCEPT;

        virtual Result GetProgress(InstallProgress* outValue) NN_NOEXCEPT = 0;
        virtual Result GetSystemUpdateTaskApplyInfo(SystemUpdateTaskApplyInfo* outValue) NN_NOEXCEPT = 0;
        virtual Result SetState(InstallProgressState state) NN_NOEXCEPT = 0;
        virtual Result SetLastResult(Result result) NN_NOEXCEPT = 0;
        virtual Result SetSystemUpdateTaskApplyInfo(SystemUpdateTaskApplyInfo info) NN_NOEXCEPT = 0;
        virtual Result Push(const void* data, size_t size) NN_NOEXCEPT = 0;
        virtual Result Count(int* outValue) NN_NOEXCEPT = 0;
        virtual Result Delete(const ContentMetaKey list[], int count) NN_NOEXCEPT = 0;
        virtual Result Cleanup() NN_NOEXCEPT = 0;

    private:
        virtual Result GetSize(size_t* outValue, int index) NN_NOEXCEPT = 0;
        virtual Result Get(int index, void* buffer, size_t size) NN_NOEXCEPT = 0;
        virtual Result Update(int index, const void* buffer, size_t size) NN_NOEXCEPT = 0;
    };

    class MemoryInstallTaskData : public InstallTaskDataBase
    {
    public:
        MemoryInstallTaskData() NN_NOEXCEPT;
        ~MemoryInstallTaskData() NN_NOEXCEPT;

        virtual Result GetProgress(InstallProgress* outValue) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetSystemUpdateTaskApplyInfo(SystemUpdateTaskApplyInfo* outValue) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetState(InstallProgressState state) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetLastResult(Result result) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetSystemUpdateTaskApplyInfo(SystemUpdateTaskApplyInfo info) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Push(const void* data, size_t size) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Count(int* outValue) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Delete(const ContentMetaKey list[], int count) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Cleanup() NN_NOEXCEPT NN_OVERRIDE;

    private:
        virtual Result GetSize(size_t* outValue, int index) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Get(int index, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Update(int index, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

        struct DataHolder : public InstallContentMeta, public util::IntrusiveListBaseNode <DataHolder> {};
        util::IntrusiveList<DataHolder, util::IntrusiveListBaseNodeTraits<DataHolder>> m_DataList;
        InstallProgressState m_State;
        Result m_LastResult;
        SystemUpdateTaskApplyInfo m_SystemUpdateTaskApplyInfo;
    };

    class FileInstallTaskData : public InstallTaskDataBase
    {
    public:
        static Result Create(const char* path, int maxEntryCount) NN_NOEXCEPT;
        Result Initialize(const char* path) NN_NOEXCEPT;

        virtual Result GetProgress(InstallProgress* outValue) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetSystemUpdateTaskApplyInfo(SystemUpdateTaskApplyInfo* outValue) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetState(InstallProgressState state) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetLastResult(Result result) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetSystemUpdateTaskApplyInfo(SystemUpdateTaskApplyInfo info) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Push(const void* data, size_t size) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Count(int* outValue) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Delete(const ContentMetaKey list[], int count) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Cleanup() NN_NOEXCEPT NN_OVERRIDE;

    private:
        struct Header
        {
            uint32_t maxEntryCount;
            uint32_t entryCount;
            int64_t lastDataOffset;
            Result lastResult;
            InstallProgressState state;
            SystemUpdateTaskApplyInfo applyInfo;
            Bit8 reserved[2];
        };

        static Header MakeInitialHeader(int maxEntryCount) NN_NOEXCEPT;

        struct EntryInfo
        {
            int64_t offset;
            int64_t size;
        };

        virtual Result GetSize(size_t* outValue, int index) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Get(int index, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Update(int index, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

        int64_t GetEntryInfoOffset(int index) NN_NOEXCEPT;
        Result GetEntryInfo(EntryInfo* outValue, int index) NN_NOEXCEPT;

        Result Write(const void* buffer, size_t bufferSize, int64_t offset) NN_NOEXCEPT;
        Result Read(void* buffer, size_t bufferSize, int64_t offset) NN_NOEXCEPT;
        Result WriteHeader() NN_NOEXCEPT;
    private:
        Header m_Header;
        char m_Path[64];
    };
}}
