﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_Types.h>
#include <nn/util/util_Uuid.h>
#include <nn/ncm/ncm_ContentId.h>
#include <nn/ncm/ncm_RightsId.h>

namespace nn { namespace ncm {

    class IContentStorage;

    /**
    * @brief プレースホルダ ID です。
    */
    struct PlaceHolderId
    {
        util::Uuid uuid;
    };

    struct NN_ALIGNAS(4) Path
    {
        char string[768];
    };

    class ContentStorage
    {
        NN_DISALLOW_COPY(ContentStorage);

    public:

        /**
        * @brief    コンストラクタです。
        */
        ContentStorage() NN_NOEXCEPT;

        /**
        * @brief    コンストラクタです。
        */
        explicit ContentStorage(sf::SharedPointer<IContentStorage> interfac) NN_NOEXCEPT;

        ContentStorage(ContentStorage&& rvalue) NN_NOEXCEPT;

        ContentStorage& operator=(ContentStorage&& rvalue) NN_NOEXCEPT;

        void swap(ContentStorage& other) NN_NOEXCEPT;

        /**
        * @brief    ユニークなプレースホルダ ID を生成します。
        *
        * @details  上位レイヤから強制中断時のクリーンアップをするために、生成されたプレースホルダ ID を
        *           上位レイヤで永続化してからプレースホルダを作成してください。
        *
        * @return   処理の結果が返ります。想定外のファイルシステムエラーが無ければ常に成功します。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - outValue にユニークなプレースホルダ ID が書き込まれている
        */
        PlaceHolderId GeneratePlaceHolderId() NN_NOEXCEPT;

        /**
        * @brief    指定された ID とサイズのコンテンツのプレースホルダを作成します。
        *
        * @details  コンテンツの配置に必要なディレクトリは自動的に作成します。
        *
        * @param[in]   placeHolderId   作成するプレースホルダの ID
        * @param[in]    contentId       作成するコンテンツの ID
        * @param[in]    size            作成するプレースホルダのサイズ
        *
        * @return   処理の結果が返ります。
        * @retval   ResultPlaceHolderAlreadyExists      指定された ID のプレースホルダが既に存在しています。
        * @retval   ResultContentAlreadyExists          指定された ID のコンテンツが既に存在します。
        * @retval   nn::fs::ResultUsableSpaceNotEnough  ファイルシステムの空き容量が不足しています。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - contentId で指定されたコンテンツの配置に必要なディレクトリが存在する
        *           - placeHolderId で指定されたプレースホルダが size の大きさで存在する
        */
        Result CreatePlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, int64_t size) NN_NOEXCEPT;

        /**
        * @brief    指定されたプレースホルダのサイズを変更します。
        *
        * @param[in]   placeHolderId   作成するプレースホルダの ID
        * @param[in]    size            作成するプレースホルダのサイズ
        *
        * @return   処理の結果が返ります。
        * @retval   ResultPlaceHolderNotFound           指定された ID のプレースホルダは存在しません。
        * @retval   fs::ResultUsableSpaceNotEnough  ファイルシステムの空き容量が不足しています。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - placeHolderId で指定されたプレースホルダが size の大きさで存在する
        */
        Result SetPlaceHolderSize(PlaceHolderId placeHolderId, int64_t size) NN_NOEXCEPT;

        /**
        * @brief    指定された ID のプレースホルダを削除します。
        *
        * @details  コンテンツの配置に必要なディレクトリは削除されません。
        *
        * @param[in]    id      削除するプレースホルダの ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultPlaceHolderNotFound           指定された ID のプレースホルダは存在しません。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - id で指定されたプレースホルダが削除されている
        */
        Result DeletePlaceHolder(PlaceHolderId id) NN_NOEXCEPT;

        /**
        * @brief    指定された ID のプレースホルダの存在を確認します。
        *
        * @param[out]   outValue    プレースホルダが存在すれば true が返ります。
        * @param[in]    id          存在を確認するプレースホルダの ID
        *
        * @return   処理の結果が返ります。想定外のファイルシステムエラーが無ければ常に成功します。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - プレースホルダが存在するとき *outValue == true、そうでないとき *outValue == false
        */
        Result HasPlaceHolder(bool* outValue, PlaceHolderId id) const NN_NOEXCEPT;

        /**
        * @brief    プレースホルダにデータを書き込みます。
        *
        * @param[in]    id          データを書き込むプレースホルダの ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultPlaceHolderNotFound           指定された ID のプレースホルダは存在しません。
        *
        * @pre
        *           - Initialize の成功
        *           - offset >= 0
        *           - offset + size <= プレースホルダのサイズ
        *           - data != nullptr
        * @post
        *           - id で指定されたプレースホルダの offset に buffer から size 分のデータが書き込まれている
        */
        Result WritePlaceHolder(PlaceHolderId id, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;

        /**
        * @brief    プレースホルダを登録して、コンテンツとして利用できるようにします。
        *
        * @param[in]    placeHolderId   登録するプレースホルダ ID
        * @param[in]    contentId       登録するコンテンツ ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultPlaceHolderNotFound           指定された ID のプレースホルダは存在しません。
        * @retval   ResultContentAlreadyExists          指定された ID のコンテンツが既に存在します。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - contentId で指定されたコンテンツが存在する
        *           - placeHolderId で指定されたプレースホルダが存在しない
        */
        Result Register(PlaceHolderId placeHolderId, ContentId contentId) NN_NOEXCEPT;

        /**
        * @brief    登録済みのコンテンツを、再度更新するためにプレースホルダに移動します。
        *
        * @param[in]    placeHolderId   登録するプレースホルダ ID
        * @param[in]    contentId       更新対象のコンテンツ ID
        * @param[in]    postContentId   更新後のコンテンツ ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultPlaceHolderAlreadyExists  指定された ID のプレースホルダがすでに存在します。
        * @retval   ResultContentNotFound           指定された ID のコンテンツが存在しません。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - contentId で指定されたコンテンツが存在しない
        *           - placeHolderId で指定されたプレースホルダが存在する
        */
        Result RevertToPlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, ContentId postContentId) NN_NOEXCEPT;

        /**
        * @brief    指定された ID のコンテンツを削除します。
        *
        * @details  コンテンツが配置されているディレクトリは削除されません。
        *
        * @param[in]    id      削除するコンテンツの ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultContentInUse                  指定された ID のコンテンツは利用中です。
        * @retval   ResultContentNotFound               指定された ID のコンテンツは存在しません。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - id で指定されたコンテンツが削除されている
        */
        Result Delete(ContentId id) NN_NOEXCEPT;

        /**
        * @brief    指定された ID のコンテンツの存在を確認します。
        *
        * @param[out]   outValue    コンテンツが存在すれば true が返ります。
        * @param[in]    id          存在を確認するコンテンツの ID
        *
        * @return   処理の結果が返ります。想定外のファイルシステムエラーが無ければ常に成功します。
        *
        * @pre
        *           - Initialize の成功
        * @post
        *           - コンテンツが存在するとき *outValue == true、そうでないとき *outValue == false
        */
        Result Has(bool* outValue, ContentId id) const NN_NOEXCEPT;

        /**
        * @brief    指定された ID のコンテンツのファイルパスを取得します。
        *
        * @details  ストレージにコンテンツが存在するかどうかは検知しません。
        *           存在を確認したい場合は Has() を利用してください。
        *
        * @param[in]    id          ファイルパスを取得するコンテンツの ID
        *
        * @return   指定された ID のコンテンツのファイルパスを示す文字列が返ります。
        *
        * @pre
        *           - Initialize の成功
        */
        void GetPath(Path* outValue, ContentId id) const NN_NOEXCEPT;

        /**
        * @brief    コンテントストレージの空き容量を取得します
        *
        * @param[out]    outValue   空き容量
        */
        Result GetFreeSpaceSize(int64_t* outValue) const NN_NOEXCEPT;

        /**
        * @brief    コンテントストレージの総容量を取得します
        *
        * @param[out]    outValue   総容量
        */
        Result GetTotalSpaceSize(int64_t* outValue) const NN_NOEXCEPT;

        void GetPlaceHolderPath(Path* outValue, PlaceHolderId id) const NN_NOEXCEPT;

        Result CleanupAllPlaceHolder() NN_NOEXCEPT;

        Result ListPlaceHolder(int* outCount, PlaceHolderId outList[], int count) const NN_NOEXCEPT;

        Result GetContentCount(int* outCount) const NN_NOEXCEPT;

        Result ListContentId(int* outCount, ContentId outList[], int count, int offset) const NN_NOEXCEPT;

        Result DisableForcibly() NN_NOEXCEPT;

        Result GetSize(int64_t* outValue, PlaceHolderId id) const NN_NOEXCEPT;

        Result GetSize(int64_t* outValue, ContentId id) const NN_NOEXCEPT;

        Result ReadContentIdFile(void* buffer, size_t size, ContentId id, int64_t offset) const NN_NOEXCEPT;

        Result GetRightsId(RightsId* outValue, PlaceHolderId id) const NN_NOEXCEPT;

        Result GetRightsId(RightsId* outValue, ContentId id) const NN_NOEXCEPT;

        Result WriteContentForDebug(ContentId id, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;

        Result FlushPlaceHolder() NN_NOEXCEPT;

        Result RepairInvalidFileAttribute() NN_NOEXCEPT;

    private:
        sf::SharedPointer<IContentStorage> m_Interface;
    };
}}
