﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <utility>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_IContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_ProgramId.h>

namespace nn { namespace ncm {

    class IContentMetaDatabase;

    typedef sf::SharedPointer<IContentMetaDatabase> ContentMetaDatabaseShared;

    struct ListCount
    {
        int listed;
        int total;
    };

    /**
    * @brief    コンテンツメタを保存・検索・読み出しするためのデータベースです。
    */
    class ContentMetaDatabase
    {
        NN_DISALLOW_COPY(ContentMetaDatabase);

    public:
        static const ApplicationId  AnyApplicationId;
        /**
        * @brief    コンストラクタです。
        */
        explicit ContentMetaDatabase() NN_NOEXCEPT;

        explicit ContentMetaDatabase(sf::SharedPointer<IContentMetaDatabase> interfac) NN_NOEXCEPT;

        ContentMetaDatabase(ContentMetaDatabase&& rvalue) NN_NOEXCEPT;

        ContentMetaDatabase& operator=(ContentMetaDatabase&& rvalue) NN_NOEXCEPT;

        void swap(ContentMetaDatabase& other) NN_NOEXCEPT;

        /**
        * @brief    任意のコンテンツメタを設定します。
        *
        * @return   処理の結果が返ります。想定外のファイルシステムエラーが無ければ常に成功します。
        *
        * @post
        *           - データベースファイルにメタ情報が永続化される
        */
        Result Set(const ContentMetaKey& key, const void* buffer, size_t size) NN_NOEXCEPT;

        /**
        * @brief    システムプログラムのメタ情報を削除します。
        *
        * @return   処理の結果が返ります。
        * @retval   ResultContentMetaNotFound   メタ情報が存在しません。
        *
        * @post
        *           - GetSystemProgramMeta でメタ情報が取得できない
        *           - データベースファイルからメタ情報が削除される
        */
        Result Remove(SystemProgramId id, uint32_t version) NN_NOEXCEPT;

        /**
        * @brief    システムデータのメタ情報を削除します。
        *
        * @return   処理の結果が返ります。
        * @retval   ResultContentMetaNotFound   メタ情報が存在しません。
        *
        * @post
        *           - GetSystemDataMeta でメタ情報が取得できない
        *           - データベースファイルからメタ情報が削除される
        */
        Result Remove(SystemDataId id, uint32_t version) NN_NOEXCEPT;

        /**
        * @brief    アプリケーションのメタ情報を削除します。
        *
        * @return   処理の結果が返ります。
        * @retval   ResultContentMetaNotFound   メタ情報が存在しません。
        *
        * @post
        *           - GetApplicationMeta でメタ情報が取得できない
        *           - データベースファイルからメタ情報が削除される
        */
        Result Remove(ApplicationId id, uint32_t version) NN_NOEXCEPT;

        /**
        * @brief    プログラムのコンテンツ ID を返します。
        *
        * @return   処理の結果が返ります。
        * @retval   ResultContentMetaNotFound       メタ情報が存在しません。
        * @retval   ResultProgramContentNotFound    対象のメタ情報にプログラムコンテンツが存在しません。
        *
        * @post
        *           - outValue に指定されたアプリケーションのメタ情報が取得されている
        */
        Result GetProgram(ContentId* outValue, ProgramId id, uint32_t version) const NN_NOEXCEPT;

        /**
        * @brief    プログラムのコンテンツ ID を返します。
        *
        * @return   処理の結果が返ります。
        * @retval   ResultContentMetaNotFound   メタ情報が存在しません。
        * @retval   ResultProgramContentNotFound    対象のメタ情報にプログラムコンテンツが存在しません。
        *
        * @post
        *           - outValue に指定されたアプリケーションのメタ情報が取得されている
        */
        Result GetLatestProgram(ContentId* outValue, ProgramId id) const NN_NOEXCEPT;

        Result GetLatestData(ContentId* outValue, DataId id) const NN_NOEXCEPT;

        Result GetControl(ContentId* outValue, ApplicationId id, uint32_t version) const NN_NOEXCEPT;

        Result GetLatestControl(ContentId* outValue, ApplicationId id) const NN_NOEXCEPT;

        Result GetHtmlDocument(ContentId* outValue, ApplicationId id, uint32_t version) const NN_NOEXCEPT;

        Result GetLatestHtmlDocument(ContentId* outValue, ApplicationId id) const NN_NOEXCEPT;

        Result GetLegalInformation(ContentId* outValue, ApplicationId id, uint32_t version) const NN_NOEXCEPT;

        Result GetLatestLegalInformation(ContentId* outValue, ApplicationId id) const NN_NOEXCEPT;

        Result GetContentIdByType(ContentId* outValue, const ContentMetaKey& key, ContentType type) const NN_NOEXCEPT;

        Result GetContentIdByTypeAndIdOffset(ContentId* outValue, const ContentMetaKey& key, ContentType type, uint8_t idOffset) const NN_NOEXCEPT;

        ListCount ListApplication(ApplicationContentMetaKey outValue[], int count) const NN_NOEXCEPT;

        ListCount ListContentMeta(ContentMetaKey outValue[], int count, ContentMetaType type = ContentMetaType::Unknown, ApplicationId appId = AnyApplicationId, Bit64 min = 0x0, Bit64 max = 0xffffffffffffffff, ContentInstallType installType = ContentInstallType::Full) const NN_NOEXCEPT;

        Result ListContentInfo(int* outCount, ContentInfo outList[], int count, const ContentMetaKey& key, int offset) const NN_NOEXCEPT;

        Result ListContentMetaInfo(int* outCount, ContentMetaInfo outList[], int count, const ContentMetaKey& key, int offset) const NN_NOEXCEPT;

        Result GetLatest(ContentMetaKey* outValue, Bit64 id) const NN_NOEXCEPT;

        Result Remove(const ContentMetaKey key) NN_NOEXCEPT;

        Result Has(bool* outValue, const ContentMetaKey& key) const NN_NOEXCEPT;

        Result HasAll(bool* outValue, const ContentMetaKey list[], int count) const NN_NOEXCEPT;

        Result HasContent(bool* outValue, const ContentMetaKey& key, const ContentId& contentId) const NN_NOEXCEPT;

        Result Get(size_t* outSize, void* outValue, size_t bufferSize, const ContentMetaKey& contentMetaKey) const NN_NOEXCEPT;

        Result GetSize(size_t* outSize, const ContentMetaKey& key) const NN_NOEXCEPT;

        Result GetRequiredSystemVersion(uint32_t* out, const ContentMetaKey& key) const NN_NOEXCEPT;

        Result GetPatchId(PatchId* out, const ContentMetaKey& key) const NN_NOEXCEPT;

        Result GetAttributes(Bit8* outAttributes, const ContentMetaKey& key) const NN_NOEXCEPT;

        Result LookupOrphanContent(bool* outList, const ContentId contentList[], int count) const NN_NOEXCEPT;

        Result Commit() NN_NOEXCEPT;

        Result DisableForcibly() NN_NOEXCEPT;

        Result GetRequiredApplicationVersion(uint32_t* outValue, const ContentMetaKey& key) const NN_NOEXCEPT;

    private:
        ContentMetaDatabaseShared m_Interface;
    };
}}
