﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_Optional.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_ContentInfo.h>
#include <nn/ncm/ncm_ContentInfoData.h>
#include <nn/ncm/ncm_ContentInfoUtil.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ncm/ncm_Result.h>

namespace nn { namespace ncm {

    enum ContentMetaAttribute
    {
        ContentMetaAttribute_None = 0,
        ContentMetaAttribute_IncludesExFatDriver = 1 << 0,
        ContentMetaAttribute_Rebootless = 1 << 1,
    };

    struct ContentMetaInfo
    {
        Bit64                   id;
        uint32_t                version;
        ContentMetaType         type;
        Bit8                    attributes;
        Bit8                    reserved[2];

        static ContentMetaInfo Make(Bit64 id, uint32_t version, ContentMetaType type, Bit8 attributes) NN_NOEXCEPT
        {
            ContentMetaInfo info = { id, version, type, attributes };
            return info;
        }

        ncm::ContentMetaKey ToKey() const NN_NOEXCEPT
        {
            ContentMetaKey key = { id, version, type};
            return key;
        }
    };

    // INFO: 最終的に ContentMetaDatabase に格納される状態のヘッダ
    //       インストール時にのみ使われる情報が削られている状態
    struct ContentMetaHeader
    {
        uint16_t                extendedHeaderSize; // 0 なら拡張ヘッダなし
        uint16_t                contentCount;
        uint16_t                contentMetaCount;
        Bit8                    attributes;
        Bit8                    reserved[1];
    };

    // INFO: パッケージングされた状態のヘッダ
    //       AuthoringTool で生成した状態はこれ
    //       Reserve が付いているものは、PackagedContentMeta の状態では 0 が書き込まれているが、
    //       InstallContentMeta として利用する段階で値を書き込めることを想定している
    struct PackagedContentMetaHeader
    {
        Bit64                   id;
        uint32_t                version;
        ContentMetaType         type;
        Bit8                    reserved1[1];
        uint16_t                extendedHeaderSize; // 0 なら拡張ヘッダなし
        uint16_t                contentCount;
        uint16_t                contentMetaCount;
        Bit8                    attributes;
        Bit8                    storageIdReserve;
        ContentInstallType      installTypeReserve; // インストールの種類 (full / meta-only)
        bool                    committedReserve;
        uint32_t                requiredDownloadSystemVersion;
        Bit8                    reserved3[4];
    };

    // INFO: インストール時に使う状態のヘッダ
    //       このヘッダがインストールコンテキストとして保存される
    typedef PackagedContentMetaHeader InstallContentMetaHeader;

    struct ApplicationMetaExtendedHeader
    {
        PatchId     patchId;
        uint32_t    requiredSystemVersion;
        Bit8        reserved[4];
    };

    struct AddOnContentMetaExtendedHeader
    {
        ApplicationId applicationId;
        uint32_t      requiredApplicationVersion;
        Bit8          reserved[4];
    };

    struct PatchMetaExtendedHeader
    {
        ApplicationId applicationId;
        uint32_t    requiredSystemVersion;
        uint32_t    extendedDataSize;
        Bit8        reserved[8];      // 将来の拡張用
    };
    struct DeltaMetaExtendedHeader
    {
        ApplicationId applicationId;
        uint32_t      extendedDataSize;
        Bit8          reserved[4];
    };

    template<class ContentMetaHeaderT, class ContentInfoT>
    class ContentMetaAccessorBase
    {
    public:
        const void* GetData() const NN_NOEXCEPT
        {
            return m_Data;
        }

        size_t GetSize() const NN_NOEXCEPT
        {
            return m_Size;
        }

        ContentMetaKey GetKey() const NN_NOEXCEPT
        {
            auto header = GetHeader();
            ContentMetaKey key = { header->id, header->version, header->type, header->installTypeReserve };
            return key;
        }

        const ContentMetaHeaderT* GetHeader() const NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(m_IsHeaderOk);

            return reinterpret_cast<const ContentMetaHeaderT*>(m_Data);
        }

        ContentMetaHeaderT* GetWritableHeader() NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(m_IsHeaderOk);

            return reinterpret_cast<ContentMetaHeaderT*>(m_Data);
        }

        template<typename ExtendedHeaderT>
        const ExtendedHeaderT* GetExtendedHeader() const NN_NOEXCEPT
        {
            return reinterpret_cast<const ExtendedHeaderT*>(GetExtendedHeaderAddress());
        }
        const void* GetExtendedData() const NN_NOEXCEPT
        {
            return reinterpret_cast<const void*>(GetExtendedDataAddress());
        }

        size_t GetExtendedDataSize() const NN_NOEXCEPT
        {
            ContentMetaType type = GetHeader()->type;
            switch(type)
            {
            case ContentMetaType::Patch:
                return GetExtendedHeader<PatchMetaExtendedHeader>()->extendedDataSize;
            case ContentMetaType::Delta:
                return GetExtendedHeader<DeltaMetaExtendedHeader>()->extendedDataSize;
            default:
                return 0;
            }
        }

        int CountContent() const NN_NOEXCEPT
        {
            return static_cast<int>(GetHeader()->contentCount);
        }

        int CountContentMeta() const NN_NOEXCEPT
        {
            return static_cast<int>(GetHeader()->contentMetaCount);
        }

        const ContentInfoT* GetContentInfo(int index) const NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(index < CountContent());

            return GetWritableContentInfo(index);
        }

        const ContentInfoT* GetContentInfo(ContentType type) const NN_NOEXCEPT
        {
            return GetWritableContentInfo(type);
        }

        const ContentInfoT* GetContentInfo(ContentType type, uint8_t idOffset) const NN_NOEXCEPT
        {
            return GetWritableContentInfo(type, idOffset);
        }

        const ContentMetaInfo* GetContentMetaInfo(int index) const NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(index < CountContentMeta());

            return reinterpret_cast<ContentMetaInfo*>(GetContentMetaInfoAddress(index));
        }

        bool HasContent(const ContentId& id) const NN_NOEXCEPT
        {
            for (int i = 0; i < CountContent(); i++)
            {
                if (id == GetContentInfo(i)->GetId())
                {
                    return true;
                }
            }

            return false;
        }

        const Hash* GetDigest() const NN_NOEXCEPT
        {
            return reinterpret_cast<Hash*>(GetDigestAddress());
        }

        util::optional<ApplicationId> GetApplicationId() const NN_NOEXCEPT
        {
            auto key = ContentMetaKey::Make(
                GetHeader()->id,
                GetHeader()->version,
                GetHeader()->type
            );
            return GetApplicationId(key);
        }

        util::optional<ApplicationId> GetApplicationId(ContentMetaKey& key) const NN_NOEXCEPT
        {
            switch (key.type)
            {
            case ContentMetaType::Application:
                {
                    ncm::ApplicationId applicationId = { key.id };
                    return applicationId;
                }
            case ContentMetaType::AddOnContent: return GetExtendedHeader<AddOnContentMetaExtendedHeader>()->applicationId;
            case ContentMetaType::Patch: return GetExtendedHeader<PatchMetaExtendedHeader>()->applicationId;
            case ContentMetaType::Delta: return GetExtendedHeader<DeltaMetaExtendedHeader>()->applicationId;
            default: return util::nullopt;
            }
        }

    protected:
        ContentMetaAccessorBase(const void* data, size_t size) NN_NOEXCEPT : m_IsHeaderOk(true), m_Data(const_cast<void*>(data)), m_Size(size){}

        ContentMetaAccessorBase(void* data, size_t size) NN_NOEXCEPT : m_IsHeaderOk(false), m_Data(data), m_Size(size){}

        void* GetWritableData() NN_NOEXCEPT
        {
            return m_Data;
        }

        uintptr_t GetExtendedHeaderAddress() const NN_NOEXCEPT
        {
            return reinterpret_cast<uintptr_t>(m_Data) + sizeof(ContentMetaHeaderT);
        }

        uintptr_t GetContentInfoHeadAddress() const NN_NOEXCEPT
        {
            return GetExtendedHeaderAddress() + GetHeader()->extendedHeaderSize;
        }

        uintptr_t GetContentInfoAddress(int index) const NN_NOEXCEPT
        {
            return GetContentInfoHeadAddress() + sizeof(ContentInfoT) * index;
        }

        uintptr_t GetContentMetaInfoHeadAddress() const NN_NOEXCEPT
        {
            return GetContentInfoAddress(CountContent());
        }

        uintptr_t GetContentMetaInfoAddress(int index) const NN_NOEXCEPT
        {
            return GetContentMetaInfoHeadAddress() + sizeof(ContentMetaInfo) * index;
        }

        uintptr_t GetExtendedDataAddress() const NN_NOEXCEPT
        {
            return GetContentMetaInfoAddress(CountContentMeta());
        }

        uintptr_t GetDigestAddress() const NN_NOEXCEPT
        {
            return GetExtendedDataAddress() + GetExtendedDataSize();
        }

        void WriteHeader(Bit64 id, uint32_t version, ContentMetaType type, Bit8 attributes, uint32_t requiredDownloadSystemVersion, int contentCount, int contentMetaCount) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(!m_IsHeaderOk);
            NN_ABORT_UNLESS(GetSize() >= CalculateSize(type, contentCount, contentMetaCount, 0)); // extendedData サイズが分からないので、とりあえず 0 としておく

            ContentMetaHeaderT header = {};
            header.id = id;
            header.version = version;
            header.type = type;
            header.attributes = attributes;
            header.extendedHeaderSize = static_cast<uint16_t>(GetExtendedHeaderSize(type));
            header.contentCount = static_cast<uint16_t>(contentCount);
            header.contentMetaCount = static_cast<uint16_t>(contentMetaCount);
            header.requiredDownloadSystemVersion = requiredDownloadSystemVersion;

            std::memcpy(GetWritableData(), &header, sizeof(header));

            m_IsHeaderOk = true;
        }

        template<typename ExtendedHeaderT>
        void WriteExtendedHeader(const ExtendedHeaderT& data) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(m_IsHeaderOk);

            std::memcpy(reinterpret_cast<void*>(GetExtendedHeaderAddress()), &data, sizeof(ExtendedHeaderT));
        }
        void WriteExtendedData(const void* data, size_t size) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(m_IsHeaderOk);

            std::memcpy(reinterpret_cast<void*>(GetExtendedDataAddress()), data, size);
        }

        void WriteContentInfo(const ContentInfoT& contentInfo, int index) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(static_cast<uint16_t>(index) < GetHeader()->contentCount);

            std::memcpy(reinterpret_cast<void*>(GetContentInfoAddress(index)), &contentInfo, sizeof(contentInfo));
        }

        void WriteContentMetaInfo(const ContentMetaInfo& contentMetaInfo, int index) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(static_cast<uint16_t>(index) < GetHeader()->contentMetaCount);

            std::memcpy(reinterpret_cast<void*>(GetContentMetaInfoAddress(index)), &contentMetaInfo, sizeof(contentMetaInfo));
        }

        void WriteDigest(const Hash& digest) NN_NOEXCEPT
        {
            std::memcpy(reinterpret_cast<void*>(GetDigestAddress()), &digest, sizeof(digest));
        }

        ContentInfoT* GetWritableContentInfo(int index) const NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(index < CountContent());

            return reinterpret_cast<ContentInfoT*>(GetContentInfoAddress(index));
        }

        ContentInfoT* GetWritableContentInfo(ContentType type) const NN_NOEXCEPT
        {
            ContentInfoT* resultInfo = nullptr;
            for (int i = 0; i < CountContent(); i++)
            {
                auto info = GetWritableContentInfo(i);
                if (info->GetType() == type)
                {
                    if (resultInfo)
                    {
                        resultInfo = (info->GetIdOffset() < resultInfo->GetIdOffset()) ? info : resultInfo;
                    }
                    else
                    {
                        resultInfo = info;
                    }
                }
            }

            return resultInfo;
        }

        ContentInfoT* GetWritableContentInfo(ContentType type, uint8_t idOffset) const NN_NOEXCEPT
        {
            for (int i = 0; i < CountContent(); i++)
            {
                auto info = GetWritableContentInfo(i);
                if (info->GetType() == type && info->GetIdOffset() == idOffset)
                {
                    return info;
                }
            }

            return nullptr;
        }

        template<class NewHeaderT, class NewContentInfoT>
        static size_t RecalculateSizeTemplate(size_t extendedHeaderSize, int contentCount, int contentMetaCount, size_t extendedDataSize, bool hasDigest) NN_NOEXCEPT
        {
            return sizeof(NewHeaderT) + extendedHeaderSize + sizeof(NewContentInfoT) * contentCount + sizeof(ContentMetaInfo) * contentMetaCount + extendedDataSize + (hasDigest ? sizeof(Hash) : 0);
        }

        static size_t CalculateSize(ContentMetaType type, int contentCount, int contentMetaCount, size_t extendedDataSize, bool hasDigest) NN_NOEXCEPT
        {
            return RecalculateSizeTemplate<ContentMetaHeaderT, ContentInfoT>(GetExtendedHeaderSize(type), contentCount, contentMetaCount, extendedDataSize, hasDigest);
        }

        static size_t CalculateSize(ContentMetaType type, int contentCount, int contentMetaCount, size_t extendedDataSize) NN_NOEXCEPT
        {
            return RecalculateSizeTemplate<ContentMetaHeaderT, ContentInfoT>(GetExtendedHeaderSize(type), contentCount, contentMetaCount, extendedDataSize, false);
        }

        int64_t CalculateContentRequiredSize() const NN_NOEXCEPT
        {
            int64_t requiredSize = 0;
            for (int i = 0; i < CountContent(); i++)
            {
                requiredSize += CalculateRequiredSize(GetContentInfo(i)->info.GetSize());
            }

            return requiredSize;
        }

        void SetStorageId(StorageId storageId) NN_NOEXCEPT
        {
            GetWritableHeader()->storageIdReserve = static_cast<Bit8>(storageId);
        }

        StorageId GetStorageId() const NN_NOEXCEPT
        {
            return static_cast<StorageId>(GetHeader()->storageIdReserve);
        }

    private:
        bool    m_IsHeaderOk;
        void*   m_Data;
        const size_t  m_Size;

        static size_t GetExtendedHeaderSize(ContentMetaType type) NN_NOEXCEPT
        {
            switch (type)
            {
            case ContentMetaType::Application: return sizeof(ApplicationMetaExtendedHeader);
            case ContentMetaType::AddOnContent: return sizeof(AddOnContentMetaExtendedHeader);
            case ContentMetaType::Patch: return sizeof(PatchMetaExtendedHeader);
            case ContentMetaType::Delta: return sizeof(DeltaMetaExtendedHeader);
            default: return 0;
            }
        }
    };

    class ContentMetaReader : public ContentMetaAccessorBase<ContentMetaHeader, ContentInfo>
    {
    public:
        ContentMetaReader(const void* data, size_t size) NN_NOEXCEPT : ContentMetaAccessorBase(data, size){}

        using ContentMetaAccessorBase::CalculateSize;
    };

    class PackagedContentMetaReader : public ContentMetaAccessorBase<PackagedContentMetaHeader, PackagedContentInfo>
    {
    public:
        PackagedContentMetaReader(const void* data, size_t size) NN_NOEXCEPT : ContentMetaAccessorBase(data, size) {}

        size_t CalculateConvertInstallContentMetaSize() const NN_NOEXCEPT;
        size_t CalculateConvertContentMetaSize() const NN_NOEXCEPT;

        void ConvertToInstallContentMeta(void* outValue, size_t size, const InstallContentInfo& meta) const NN_NOEXCEPT;
        void ConvertToContentMeta(void* outValue, size_t size, const ContentInfo& meta) const NN_NOEXCEPT;

        Result CalculateConvertFragmentOnlyInstallContentMetaSize(size_t* outSize, uint32_t sourceVersion) const NN_NOEXCEPT;
        Result ConvertToFragmentOnlyInstallContentMeta(void* outValue, size_t size, const InstallContentInfo& meta, uint32_t sourceVersion) const NN_NOEXCEPT;

        int16_t CountDeltaFragments() const NN_NOEXCEPT;

        static size_t CalculateSize(ContentMetaType type, int contentCount, int contentMetaCount, size_t extendedDataSize) NN_NOEXCEPT
        {
            return ContentMetaAccessorBase::CalculateSize(type, contentCount, contentMetaCount, extendedDataSize, true);
        }
    };

    class InstallContentMetaReader : public ContentMetaAccessorBase<InstallContentMetaHeader, InstallContentInfo>
    {
    public:
        InstallContentMetaReader(const void* data, size_t size) NN_NOEXCEPT : ContentMetaAccessorBase(data, size){}

        using ContentMetaAccessorBase::CalculateSize;
        using ContentMetaAccessorBase::CalculateContentRequiredSize;
        using ContentMetaAccessorBase::GetStorageId;

        size_t CalculateConvertSize() const NN_NOEXCEPT;

        void ConvertToContentMeta(void* outValue, size_t size) const NN_NOEXCEPT;
    };

    class InstallContentMetaWriter : public ContentMetaAccessorBase<InstallContentMetaHeader, InstallContentInfo>
    {
    public:
        InstallContentMetaWriter(const void* data, size_t size) NN_NOEXCEPT : ContentMetaAccessorBase(data, size){}

        using ContentMetaAccessorBase::CalculateSize;
        using ContentMetaAccessorBase::CalculateContentRequiredSize;
        using ContentMetaAccessorBase::GetWritableContentInfo;
        using ContentMetaAccessorBase::SetStorageId;
    };

    // INFO: AuthoringTool 用
    class PackagedContentMetaWriter : public ContentMetaAccessorBase<PackagedContentMetaHeader, PackagedContentInfo>
    {
    public:
        PackagedContentMetaWriter(void* buffer, size_t size) NN_NOEXCEPT : ContentMetaAccessorBase(buffer, size) {}

        using ContentMetaAccessorBase::WriteHeader;
        using ContentMetaAccessorBase::WriteExtendedHeader;
        using ContentMetaAccessorBase::WriteContentInfo;
        using ContentMetaAccessorBase::WriteContentMetaInfo;
        using ContentMetaAccessorBase::WriteExtendedData;
        using ContentMetaAccessorBase::WriteDigest;
        using ContentMetaAccessorBase::GetWritableContentInfo;
        using ContentMetaAccessorBase::GetDigestAddress;

        static size_t CalculateSize(ContentMetaType type, int contentCount, int contentMetaCount, size_t extendedDataSize) NN_NOEXCEPT
        {
            return ContentMetaAccessorBase::CalculateSize(type, contentCount, contentMetaCount, extendedDataSize, true);
        }
    };
}}
