﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/mbuf/detail/mbuf_TypesPrivate.h>

namespace nn { namespace mbuf
{
    /**
     @brief         mbuf の確保方法です。
     */
    enum MbufAllocationMode
    {
        MbufAllocationMode_DontWait = 0,           //!< 即座に確保できない場合はエラーを返します。
        MbufAllocationMode_Wait     = 1,           //!< 確保できるまでブロックします。
    };

    /**
     @brief         mbuf におけるパケットの扱いに関するフラグです。
     */
    enum MbufPacketHeaderFlag
    {
        MbufPacketHeaderFlag_Broadcast = 0x0200,   //!< Broadcast パケットです。
        MbufPacketHeaderFlag_Multicast = 0x0400,   //!< Multicast パケットです。
        MbufPacketHeaderFlag_Loopback  = 0x0800,   //!< Loopback パケットです。
        MbufPacketHeaderFlag_FreeList  = 0x8000
    };

    //! @ref MbufCopydata() で mbuf チェインの最後までコピーするために指定します。
    const size_t MbufCopyall = 1000000000;

    //! mbuf のヘッダサイズです。
    const size_t MbufHeaderSize = sizeof(void*) * 3 + sizeof(Bit16) * 8 +
                                  sizeof(detail::MacAddress) * 2;

    //! mbuf の最小サイズです。
    const size_t MbufUnitSizeMin = 64;

    //! mbuf の最大サイズです。
    const size_t MbufUnitSizeMax = 65535;

    //! mbuf のデータ領域の最小サイズです。
    const size_t MbufDataSizeMin = MbufUnitSizeMin - MbufHeaderSize;

    //! mbuf のデータ領域の最大サイズです。
    const size_t MbufDataSizeMax = MbufUnitSizeMax - MbufHeaderSize;

    //! mbuf プールの最大数です。
    const int PoolCountMax = 16;

    //! 1 つの mbuf プールの最小容量 (mbuf の数) です。
    const int PoolCapacityMin = 4;

    //! 1 つの mbuf プールの最大容量 (mbuf の数) です。
    const int PoolCapacityMax = 65535;

    //! type の最小値です。
    const int TypeMin = 0;

    //! type の最大値です。
    const int TypeMax = 65535;

    /**
     * @brief       mbuf のデータ構造です。
     */
    struct Mbuf
    {
        struct Mbuf*       _pNext;                 //!< 次のクラスタへのポインタ
        struct Mbuf*       _pPrev;                 //!< 前のクラスタへのポインタ
        void*              _pGeneral;              //!< 任意の用途に利用出来るポインタ
        uint16_t           _len;                   //!< data 領域に格納されているデータ長
        uint16_t           _capacity;              //!< data 領域に格納できる最大長
        uint16_t           _top;                   //!< data 領域の先頭から実際のデータまでのオフセット値
        Bit16              _flags;                 //!< @ref MbufPacketHeaderFlag の論理和
        Bit16              _protocol;              //!< Protocol id（ソケットおよび関連モジュール用）
        Bit16              _owner;                 //!< オーナーモジュール
        Bit16              _type;                  //!< 用途
        Bit16              _pool;                  //!< どのバッファプールに確保されているかを示す識別子
        detail::MacAddress _dst;                   //!< 宛先 MAC アドレス
        detail::MacAddress _src;                   //!< 送信元 MAC アドレス
        Bit8               _data[MbufDataSizeMax]; //!< データ領域
    };

}} // end of namespace nn::mbuf
