﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_Types.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentMetaKey.h>

#include <nn/lr/lr_Path.h>

namespace nn { namespace lr {

    class ILocationResolver;
    class LocationResolver
    {
        NN_DISALLOW_COPY(LocationResolver);

    public:

        /**
        * @brief    コンストラクタです。
        */
        LocationResolver() NN_NOEXCEPT{}

        /**
        * @brief    コンストラクタです。
        */
        explicit LocationResolver(sf::SharedPointer<ILocationResolver> interfac) NN_NOEXCEPT : m_Interface(interfac) {}

        LocationResolver(LocationResolver&& rvalue) NN_NOEXCEPT
        {
            m_Interface = std::move(rvalue.m_Interface);
        }

        LocationResolver& operator=(LocationResolver&& rvalue) NN_NOEXCEPT
        {
            LocationResolver(std::move(rvalue)).swap(*this);

            return *this;
        }

        void swap(LocationResolver& other) NN_NOEXCEPT
        {
            std::swap(m_Interface, other.m_Interface);
        }

        /**
        * @brief    指定されたプログラム ID で識別されるプログラムのプログラムパスを解決して取得します。
        *
        * @detail   取得したプログラムパスを fs でマウントすることで、プログラムに含まれるデータを読み出す事ができます。
        *           プログラムパスには、プログラムが含まれているファイルシステムの形式を示す識別子が含まれます。
        *           （例：nca, ディレクトリ）
        *           fs はプログラムパスを解釈して、ユーザに適切にファイルシステムを提供します。
        *
        * @param[out]   outValue    プログラムパスを取得するバッファ
        * @param[in]    id          プログラムパスを取得するプログラムの ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultProgramNotFound   指定されたプログラムが存在しません。
        */
        Result ResolveProgramPath(Path* outValue, ncm::ProgramId id) NN_NOEXCEPT;

        /**
        * @brief    指定されたプログラム ID で解決されるプログラムパスの設定を上書きします。
        *
        * @detail   上書き可能な設定の個数は固定個（個数未定）です。
        *
        * @param[in]    id              上書き設定対象のプログラムの ID
        * @param[in]    redirectedPath  上書き設定するパス
        *
        * @pre
        *           - 開発環境で実行される
        */
        void RedirectProgramPath(ncm::ProgramId id, const Path& redirectedPath) NN_NOEXCEPT;

        /**
        * @brief    アプリケーションのプログラムパスの設定を上書きします。
        *
        * @detail   上書き可能な設定の個数は固定個（個数未定）です。
        *
        * @param[in]    id              上書き設定対象のアプリケーションの ID
        * @param[in]    redirectedPath  上書き設定するパス
        *
        * @pre
        *           - 開発環境で実行される
        */
        void RedirectApplicationProgramPath(ncm::ProgramId id, const Path& redirectedPath) NN_NOEXCEPT;

        /**
        * @brief    指定されたデータ ID で識別されるデータのデータパスを解決して取得します。
        *
        * @detail   取得したデータパスを fs でマウントすることで、データを読み出す事ができます。
        *           データパスには、データが含まれているファイルシステムの形式を示す識別子が含まれます。
        *           （例：nca, ディレクトリ）
        *           fs はデータパスを解釈して、ユーザに適切にファイルシステムを提供します。
        *
        * @param[out]   outValue    データパスを取得するバッファ
        * @param[in]    id          データパスを取得するデータの ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultDataNotFound   指定されたデータが存在しません。
        */
        Result ResolveDataPath(Path* outValue, ncm::DataId id) NN_NOEXCEPT;

        /**
        * @brief    指定されたアプリケーション ID で識別されるアプリケーションの管理データパスを解決して取得します。
        *
        * @detail   取得したパスを fs でマウントすることで、管理データに含まれるデータを読み出す事ができます。
        *           パスには、管理データが含まれているファイルシステムの形式を示す識別子が含まれます。
        *           （例：nca, ディレクトリ）
        *           fs はパスを解釈して、ユーザに適切にファイルシステムを提供します。
        *
        * @param[out]   outValue    管理データパスを取得するバッファ
        * @param[in]    id          管理データパスを取得するプログラムの ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultApplicationNotFound   指定されたプログラムが存在しません。
        */
        Result ResolveApplicationControlPath(Path* outValue, ncm::ProgramId id) NN_NOEXCEPT;

        /**
        * @brief    指定されたアプリケーション ID で解決されるアプリケーションの管理データパスの設定を上書きします。
        *
        * @detail   上書き可能な設定の個数は固定個（個数未定）です。
        *
        * @param[in]    id              上書き設定対象のアプリケーションの ID
        * @param[in]    redirectedPath  上書き設定するパス
        *
        * @pre
        *           - 開発環境で実行される
        */
        void RedirectApplicationControlPath(ncm::ProgramId id, const Path& redirectedPath) NN_NOEXCEPT;

        /**
        * @brief    指定されたアプリケーション ID で識別されるアプリケーションのアプリケーションドキュメントパスを解決して取得します。
        *
        * @detail   取得したパスを fs でマウントすることで、アプリケーションドキュメントに含まれるデータを読み出す事ができます。
        *           パスには、アプリケーションドキュメントが含まれているファイルシステムの形式を示す識別子が含まれます。
        *           （例：nca, ディレクトリ）
        *           fs はパスを解釈して、ユーザに適切にファイルシステムを提供します。
        *
        * @param[out]   outValue    アプリケーションドキュメントパスを取得するバッファ
        * @param[in]    id          アプリケーションドキュメントパスを取得するアプリケーションの ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultApplicationNotFound   指定されたアプリケーションが存在しません。
        */
        Result ResolveApplicationHtmlDocumentPath(Path* outValue, ncm::ProgramId id) NN_NOEXCEPT;

        /**
        * @brief    指定されたアプリケーション ID で解決されるアプリケーションのアプリケーションドキュメントパスの設定を上書きします。
        *
        * @detail   上書き可能な設定の個数は固定個（個数未定）です。
        *
        * @param[in]    id              上書き設定対象のアプリケーションの ID
        * @param[in]    redirectedPath  上書き設定するパス
        *
        * @pre
        *           - 開発環境で実行される
        */
        void RedirectApplicationHtmlDocumentPath(ncm::ProgramId id, const Path& redirectedPath) NN_NOEXCEPT;

        /**
        * @brief    指定されたアプリケーション ID で識別されるアプリケーションのソフトリーガル情報パスを解決して取得します。
        *
        * @detail   取得したパスを fs でマウントすることで、ソフトリーガル情報に含まれるデータを読み出す事ができます。
        *           パスには、ソフトリーガル情報が含まれているファイルシステムの形式を示す識別子が含まれます。
        *           （例：nca, ディレクトリ）
        *           fs はパスを解釈して、ユーザに適切にファイルシステムを提供します。
        *
        * @param[out]   outValue    ソフトリーガル情報パスを取得するバッファ
        * @param[in]    id          ソフトリーガル情報パスを取得するアプリケーションの ID
        *
        * @return   処理の結果が返ります。
        * @retval   ResultApplicationNotFound   指定されたアプリケーションが存在しません。
        */
        Result ResolveApplicationLegalInformationPath(Path* outValue, ncm::ProgramId id) NN_NOEXCEPT;

        /**
        * @brief    指定されたアプリケーション ID で解決されるアプリケーションのソフトリーガル情報パスの設定を上書きします。
        *
        * @detail   上書き可能な設定の個数は固定個（個数未定）です。
        *
        * @param[in]    id              上書き設定対象のアプリケーションの ID
        * @param[in]    redirectedPath  上書き設定するパス
        *
        * @pre
        *           - 開発環境で実行される
        */
        void RedirectApplicationLegalInformationPath(ncm::ProgramId id, const Path& redirectedPath) NN_NOEXCEPT;

        /**
        * @brief    LocationResolver を更新します。
        *
        * @detail   内部に保持する状態を更新します。
        *
        */
        Result Refresh() NN_NOEXCEPT;

        /**
        * @brief    登録されているアプリケーションの上書き設定を削除します。
        *
        * @pre
        *           - 開発環境で実行される
        */
        Result ClearApplicationRedirection() NN_NOEXCEPT;

        /**
        * @brief    登録されている Program の上書き設定を削除します。
        *
        * @param[in]    id              上書き設定を削除するプログラムの ID
        *
        */
        Result EraseProgramRedirection(ncm::ProgramId id) NN_NOEXCEPT;

        /**
        * @brief    登録されている Control の上書き設定を削除します。
        *
        * @param[in]    id              上書き設定を削除するプログラムの ID
        *
        */
        Result EraseApplicationControlRedirection(ncm::ProgramId id) NN_NOEXCEPT;

        /**
        * @brief    登録されている HtmlDocument の上書き設定を削除します。
        *
        * @param[in]    id              上書き設定を削除するプログラムの ID
        *
        */
        Result EraseApplicationHtmlDocumentRedirection(ncm::ProgramId id) NN_NOEXCEPT;

        /**
        * @brief    登録されている LegalInformation の上書き設定を削除します。
        *
        * @param[in]    id              上書き設定を削除するプログラムの ID
        *
        */
        Result EraseApplicationLegalInformationRedirection(ncm::ProgramId id) NN_NOEXCEPT;

    private:
        sf::SharedPointer<ILocationResolver> m_Interface;
    };
}}
