﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_Directory.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace kvdb {

    struct SimpleKeyValue
    {
        static const size_t MaxKeySize = 128;
        Bit8 key[MaxKeySize];
        void* value;
        size_t keySize;
        size_t valueSize;
    };

    class SimpleKeyValueCache
    {
    public:
        Result Initialize(void* buffer, size_t bufferSize, int maxCount) NN_NOEXCEPT;
        void InvalidateAll() NN_NOEXCEPT;
        void Invalidate(const void* key, size_t keySize) NN_NOEXCEPT;

        void Put(const void* key, size_t keySize, const void* value, size_t valueSize) NN_NOEXCEPT;
        util::optional<size_t> TryGet(void* buffer, size_t bufferSize, const void* key, size_t keySize) NN_NOEXCEPT;
        util::optional<size_t> TryGetSize(const void* key, size_t keySize) NN_NOEXCEPT;
        bool Has(const void* key, size_t keySize) NN_NOEXCEPT;

    private:
        void* m_Buffer;
        size_t m_BufferSize;
        mem::StandardAllocator m_Allocator;
        SimpleKeyValue* m_List;
        int m_MaxCount;
        int m_Count{};
    };

    class FileKeyValueStore
    {
        NN_DISALLOW_COPY(FileKeyValueStore);
        NN_DISALLOW_MOVE(FileKeyValueStore);

    public:
        FileKeyValueStore() NN_NOEXCEPT{}

        typedef BoundedString<768> Path;

        Result Initialize(const char* directoryPath) NN_NOEXCEPT;

        Result Initialize(const char* directoryPath, void* cacheBuffer, size_t cacheSize, int maxCacheCount) NN_NOEXCEPT;

        Result Put(const void* key, size_t keySize, const void* value, size_t valueSize) NN_NOEXCEPT;

        Result Get(size_t* outValue, void* buffer, size_t bufferSize, const void* key, size_t keySize) NN_NOEXCEPT;

        Result GetSize(size_t* outValue, const void* key, size_t keySize) NN_NOEXCEPT;

        Result Delete(const void* key, size_t keySize) NN_NOEXCEPT;

    private:
        Path m_DirectoryPath;
        os::Mutex m_Mutex{ false };
        SimpleKeyValueCache m_Cache;
    };

}}
