﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
*   @file
*   @brief  ゲームカードドライバ API の宣言
*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/gc/gc_Result.h>
#include <nn/fs/fs_GameCard.h>

#include <nn/gc/detail/gc_Types.h>
#include <nn/gc/detail/gc_Define.h>

#include <nn/sdmmc/sdmmc_GcAsic.h>
#include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>

namespace nn { namespace gc {

/**
* @brief ドライバ内に持つワークバッファのサイズです
*/
static const size_t GcWorkBufferSize =
    SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(nn::gc::detail::GcAsicFirmwareSize)
    + SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(nn::dd::DeviceAddressSpaceMemoryRegionAlignment);

/**
* @brief ゲームカードに関連する固定値です
*/

static const size_t GcPageSize = 512;
static const size_t GcCardHeaderPageCount = 1;
static const size_t GcReservedAreaPageCount = 0x37;
static const size_t GcCertAreaPageCount = 0x40;
static const size_t GcPackageIdSize = 0x08;
static const size_t GcPartitionFsHeaderHashSize = 0x20;
static const size_t GcCardDeviceIdSize = 0x10;
static const size_t GcDeviceCertificateSize = 512;
static const size_t GcCardImageHashSize = 32;

struct GameCardStatus
{
    char partitionFsHeaderHash[GcPartitionFsHeaderHashSize];
    // （カードヘッダから取得した値）
    char packageId[GcPackageIdSize];
    // （カード初期化時に取得した値）
    uint64_t cardSize;// Byte
    uint64_t partitionFsHeaderAddress;// Byte
    uint64_t partitionFsHeaderSize;// Byte
    uint64_t normalAreaSize;// Byte
    uint64_t secureAreaSize;// Byte
    uint32_t cupVersion;
    uint64_t cupId;
    uint8_t  reserved1[4];
    uint8_t  flags;
    uint8_t  reserved2[7];
};

/**
*   @brief ゲームカードの ID セットです。
*/
struct GameCardIdSet
{
    nn::gc::detail::CardId1 id1;
    nn::gc::detail::CardId2 id2;
    nn::gc::detail::CardId3 id3;
};

//! @name ゲームカードドライバAPI
//! @{


//--------------------------------------------------------------------------
/**
*   @brief      ゲームカードドライバで使用される鍵情報を設定します。
*               Initialize よりも前に一度だけ呼ぶ必要があります。
*
*   @param[in]  encryptedKeyBuffer      暗号化された SoC 秘密鍵のバッファ
*   @param[in]  encryptedKeyBufferSize  暗号化された SoC 秘密鍵のバッファのサイズ
*   @param[in]  socCertBuffer           SoC 証明書のバッファ
*   @param[in]  socCertBufferSize       SoC 証明書のバッファのサイズ
*
*   @return     なし
*/

void PresetInternalKeys(const void* encryptedKeyBuffer, const size_t encryptedKeyBufferSize,
                        const void* socCertBuffer, const size_t socCertBufferSize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      ゲームカードドライバの初期化を行います。(sMMU 対応アプリ用)
*               事前に nn::spl の Initialize が行われている必要があります。
*
*   @param[in]  workBuffer       ゲームカードドライバに与えるワークバッファ
*   @param[in]  workBufferSize ワークバッファのサイズ
*   @param[in]  workBufferDeviceVirtualAddress  ワークバッファのデバイスアドレス
*
*   @return     なし
*/

void Initialize(void* workBuffer,const size_t workBufferSize, const nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ゲームカードドライバの終了処理を行います。
*
*   @return     なし
*/
void Finalize() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ゲームカードの電源断を行います。呼び出し状況はシャットダウン時のみを想定します。
*
*   @return     なし
*/
void PowerOffGameCard() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      バッファのデバイスアドレスを登録します。(sMMU 対応アプリ用)
*
*   @param[in]  bufferAddress   バッファのメモリアドレス
*   @param[in]  bufferSize      バッファのサイズ
*   @param[in]  bufferDeviceVirtualAddress  バッファのデバイスアドレス
*
*   @return     なし
*/
void RegisterDeviceVirtualAddress(const uintptr_t bufferAddress, const size_t bufferSize, const nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      バッファのデバイスアドレスを削除します。
*
*   @param[in]  bufferAddress   バッファのメモリアドレス
*   @param[in]  bufferSize      バッファのサイズ
*   @param[in]  bufferDeviceVirtualAddress  バッファのデバイスアドレス
*
*   @return     なし
*/
void UnregisterDeviceVirtualAddress(const uintptr_t bufferAddress, const size_t bufferSize, const nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      ゲームカード用 Bridge ASIC とのセッション構築が正しく行われたかを調べます。
*               セッション構築が終了するまでブロックします。
*
*   @return     セッション構築の結果が返ります。
*/

nn::Result GetInitializationResult() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      ゲームカードのバスを有効化します。カードが挿入されたら初めに呼んでください。
*               Activate 後にカードの抜去・再挿入があると内部でハードウェアをリセットするため復帰に時間がかかります。
*               Activate した後は Deactivate しない限り、再度 Activate を行うことはできません。
*               ゲームカード未挿入時（IsCardInserted が false の時）は処理に失敗します。
*
*   @return     処理の結果が返ります。
*/
nn::Result Activate() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      Activate 後、カードが抜去された時や、明示的に再初期化が必要な時に呼んでください。
*/
void Deactivate() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ゲームカードの初期化を行います。
*               Activate 済である必要があります。
*               ゲームカード未挿入時は処理に失敗します。
*
*   @pre
*       - Activate() が呼ばれた後である
*
*   @return     処理の結果が返ります。
*/
nn::Result SetCardToSecureMode() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ゲームカードメモリのデータを読み出します。
*               SetCardToSecureMode に成功している必要があります。
*
*   @param[in]  outDataBuffer  読み込みデータの格納先バッファ
*   @param[in]  dataBufferSize  読み込みデータの格納先バッファのサイズ
*   @param[in]  pageAddress     読み込みを開始するページアドレス
*   @param[in]  pageCount       読み込むページサイズ
*
*   @return     処理の結果が返ります。同時呼び出し可能数を超過した場合、ResultQueueFullFailure が返ります。
*/
nn::Result Read(void* outDataBuffer, const size_t dataBufferSize, const uint32_t pageAddress, const uint32_t pageCount) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ゲームカード用 Bridge ASIC をスリープモードに移行します。
*               カードが Activate されていた場合、スリープ準備として ASIC を初期化するまでブロックします。
*
*   @return     なし
*/
void PutToSleep() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ゲームカード用 Bridge ASIC をスリープモードから復帰させます。
*               スリープ前にカードが Activate や初期化されていた場合はその状態まで復帰させるトリガをかけ、ブロックせずに帰ります。
*               GC ドライバの内部スレッドが状態復帰を行っている間に次の処理 API を呼んだ場合は、状態復帰が終わるまで次の処理がブロックされます。
*
*   @return     なし
*/
void Awaken() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ゲームカードが現在挿入されているかどうかを調べます。
*
*   @return     ゲームカードが現在挿入されていれば true が返ります。
*/
bool IsCardInserted() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ゲームカードの Activate が現在有効かどうかを調べます。
*
*   @return     ゲームカードの Activate が有効なら true が返ります。
*/
bool IsCardActivationValid() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      現在挿入されているゲームカードに関連する情報を取得します。
*               Activate 済である必要があります。
*
*   @param[in]  pOutValue   取得情報の格納先変数へのポインタ
*
*   @return     処理の結果が返ります。
*/
nn::Result GetCardStatus(GameCardStatus* pOutValue) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      現在挿入されているゲームカードのデバイス固有IDを取得します。
*               格納先のバッファは GcCardDeviceIdSize 以上の長さを持つ必要があります。
*
*   @param[in]  outBuffer  読み込みデータの格納先バッファ
*   @param[in]  outBufferSize  読み込みデータの格納先バッファのサイズ
*
*   @pre
*       - SetCardToSecureMode() が呼ばれた後である
*
*   @return     処理の結果が返ります。
*/
nn::Result GetCardDeviceId(void* outBuffer, const size_t outBufferSize) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      現在挿入されているゲームカードの証明書を取得します。
*               格納先のバッファは証明書 (512 Byte) 以上の長さを持つ必要があります。
*
*   @param[in]  outBuffer  読み込みデータの格納先バッファ
*   @param[in]  outBufferSize  読み込みデータの格納先バッファのサイズ
*
*   @pre
*       - SetCardToSecureMode() が呼ばれた後である
*
*   @return     処理の結果が返ります。
*/
nn::Result GetCardDeviceCertificate(void* outBuffer, const size_t outBufferSize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      現在挿入されているゲームカードのカードイメージハッシュを取得します。
*               格納先のバッファはカードイメージハッシュに等しい長さを持つ必要があります。
*
*   @param[in]  outBuffer  読み込みデータの格納先バッファ
*   @param[in]  outBufferSize  読み込みデータの格納先バッファのサイズ
*
*   @pre
*       - ActivateCard() が呼ばれた後である
*
*   @return     処理の結果が返ります。
*/
nn::Result GetCardImageHash(void* outBuffer, const size_t outBufferSize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      現在挿入されているゲームカードの HW 的な ID 値を取得します。検査目的で使用します。
*
*   @param[in]  pOutValue   挿入されているカードの GameCardIdSet の格納先変数へのポインタ
*
*   @return     処理の結果が返ります。
*/
nn::Result GetGameCardIdSet(GameCardIdSet* pOutValue) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ゲームカード挿抜時に呼ばれるコールバック関数を登録します。
*               このコールバック関数内ではイベントをシグナルするだけとし、時間のかかる処理を実装しないでください。
*
*   @param[in]  pDetectionEventCallback  コールバックに登録する関数ポインタ
*   @param[in]  pParameter               コールバック呼び出し時に与える引数データのポインタ
*
*   @return     なし
*/
void RegisterDetectionEventCallback(nn::gc::detail::GcCallbackFunctionPointer const pDetectionEventCallback, void* const pParameter) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      ゲームカード挿抜時に呼ばれるコールバック関数を解除します。
*
*   @return     なし
*/
void UnregisterDetectionEventCallback() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      Activate をせずにカードヘッダだけを取得します。取得後はリセットしないと使えない。
*
*   @param[in]  outBuffer  読み込みデータの格納先バッファ
*   @param[in]  outBufferSize  読み込みデータの格納先バッファのサイズ
*
*   @return     処理の結果が返ります。
*/
Result GetCardHeader(void* outBuffer, const size_t outBufferSize) NN_NOEXCEPT;

Result GetErrorInfo(nn::fs::GameCardErrorReportInfo* pOutGameCardErrorReportInfo) NN_NOEXCEPT;

//! @}

}} // namespace nn::gc
