﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/dd.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/gc/gc.h>

#include <nn/gc/detail/gc_ThreadInterface.h>
#include <nn/gc/detail/gc_Util.h>

namespace nn { namespace gc {
namespace detail {


// 処理キューに関するクラス定義
class WorkClientLocker
{
    NN_DISALLOW_COPY(WorkClientLocker);

private:
    bool m_IsUsed;
    nn::os::EventType m_WorkFinishEvent;
    nn::Result m_Result;

public:
    WorkClientLocker() NN_NOEXCEPT;
    ~WorkClientLocker() NN_NOEXCEPT;

    void Reset() NN_NOEXCEPT;
    bool IsAvailable() NN_NOEXCEPT
    {
        return ! m_IsUsed;
    }
    void SetResult(nn::Result result) NN_NOEXCEPT
    {
        m_Result = result;
    }
    void Hold() NN_NOEXCEPT;
    void WaitEvent() NN_NOEXCEPT;
    void SignalEvent() NN_NOEXCEPT;
    nn::Result Release() NN_NOEXCEPT
    {
        m_IsUsed = false;
        return m_Result;
    }
};

// 処理キューを登録するスレッドセーフなリングバッファ
class AsicWorkRingBuffer : private RingBuffer<AsicWorkInfo>
{
    NN_DISALLOW_COPY(AsicWorkRingBuffer);

private:
    mutable nn::os::Mutex m_RingBufferMutex;

public:
    AsicWorkRingBuffer(AsicWorkInfo* dataBuffer, const size_t dataBufferLength) NN_NOEXCEPT;
    void Enqueue(AsicWorkInfo& data) NN_NOEXCEPT;
    void Dequeue(AsicWorkInfo* pOutValue) NN_NOEXCEPT;
    size_t GetLength() NN_NOEXCEPT;
    bool IsEmpty() NN_NOEXCEPT;
};

class AsicHandlerCore;

// ASIC 関連処理を行うキュースレッドクラス：実際の仕事は AsicHandlerCore が行う
class AsicHandler : public ThreadInterface
{
    NN_DISALLOW_COPY(AsicHandler);

public:
    static const int GcAsicWorkQueueLength = 16;    // NOTE: 外に見せるには gc ドライバ内でキューに入れるケースを考慮する必要がある（エラーハンドリングなど）
    const char* ThreadNamePtr;

private:
    // スレッド設定
    static const size_t ThreadStackSize = 32 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char m_ThreadStack[ ThreadStackSize ];

    // ワークバッファ系
    uintptr_t m_WorkBuffer;
    size_t m_WorkBufferLength;
    nn::dd::DeviceVirtualAddress m_WorkBufferDeviceVirtualAddress;

    // 処理キュー
    AsicWorkRingBuffer m_AsicWorkQueue;
    // 処理に対する結果保存・通知
    WorkClientLocker m_AsicWorkClientLocker[GcAsicWorkQueueLength];
    // 処理キューに関する排他（fs ワーカスレッド間の排他）
    mutable nn::os::Mutex m_AsicLockerMutex;

    // 上位レイヤ向けコールバック
    GcCallbackStruct m_CardDetectionEventCallback;

    // 割り込みイベント
    nn::os::EventType* m_pGpioDetectReceiveEvent;
    // 処理要求イベント
    nn::os::EventType m_WorkRequestReceiveEvent;

    // 待ち受け
    nn::os::MultiWaitHolderType m_GpioDetectEventHolder;
    nn::os::MultiWaitHolderType m_WorkRequestEventHolder;

    // AsicHandlerCore が実処理を行う
    AsicHandlerCore& m_AsicHandlerCore;

public:
    // 以下、メインスレッドから呼ばれる API（できればここにブロッキング API は置かない）
    static AsicHandler& GetInstance() NN_NOEXCEPT;
    nn::Result Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    // 挿抜時に呼ぶ上位レイヤ向けコールバック関数を登録する
    void RegisterDetectionEventCallback(const GcCallbackStruct* pDetectionEventCallback) NN_NOEXCEPT;
    void UnregisterDetectionEventCallback() NN_NOEXCEPT;
    void SetBuffer(void* workBuffer, const size_t workBufferLength, const nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress)  NN_NOEXCEPT;

    // 仕事の依頼受付窓口
    nn::Result OrderWork(AsicWork workIndex) NN_NOEXCEPT;
    nn::Result OrderWorkWithNonBlocking(AsicWork workIndex) NN_NOEXCEPT;
    nn::Result OrderWorkRead(char* outDataBuffer, const size_t bufferLength, const uint32_t pageAddress, const uint32_t pageCount) NN_NOEXCEPT;
    nn::Result OrderWorkGetCardStatus(GameCardStatus* pOutValue) NN_NOEXCEPT;
    nn::Result OrderWorkGetCardDeviceId(char* outBuffer, const size_t outBufferSize) NN_NOEXCEPT;
    nn::Result OrderWorkGetCardDeviceCertificate(char* outBuffer, const size_t outBufferSize) NN_NOEXCEPT;
    nn::Result OrderWorkGetCardImageHash(char* outBuffer, const size_t outBufferSize) NN_NOEXCEPT;
    nn::Result OrderWorkGetCardIdSet(GameCardIdSet* pOutValue) NN_NOEXCEPT;
    nn::Result OrderWorkGetCardHeader(char* outBuffer, const size_t outBufferSize) NN_NOEXCEPT;
    nn::Result OrderWorkGetErrorInfo(nn::fs::GameCardErrorReportInfo* pOutGameCardErrorInfo) NN_NOEXCEPT;
    nn::Result OrderWorkIsCardActivationValid(bool* pOutIsCardActivationValid) NN_NOEXCEPT;
    // 内部・AsicHandlerCore からのみ利用
    nn::Result OrderWorkInternalWithBlocking(AsicWork workIndex, ReadInfo* readInfo) NN_NOEXCEPT;

private:
    // 仕事の実登録処理
    nn::Result OrderWorkInternal(AsicWork workIndex, ReadInfo* readInfo, bool isNoBlock) NN_NOEXCEPT;

    // 以下、スレッド関連の自動的・内部的に呼ばれる API
    void InitializeThread() NN_NOEXCEPT;
    virtual void RunThreadFunctionImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void InitializeWaitEvents() NN_NOEXCEPT NN_OVERRIDE;
    virtual void FinalizeWaitEvents() NN_NOEXCEPT NN_OVERRIDE;

    // キュースレッドとしての API
    nn::Result HandleWorkRequest(AsicWorkInfo workInfo) NN_NOEXCEPT;
    nn::Result AllocateLocker(int* pOutIndex, WorkClientLocker** pOutLockerPtr) NN_NOEXCEPT;
    nn::Result ReturnLocker(int lockerIndex) NN_NOEXCEPT;
    bool IsLockerIndexInRange(int index) NN_NOEXCEPT;

private:
    AsicHandler() NN_NOEXCEPT;
};


} } }
