﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/utilTool/fs_BucketTreeAnalyzer.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace fssystem { namespace utilTool {

template< typename TFunc >
Result OutputBucketTreeBuildLog(
           fs::IStorage* pTableFileStorage,
           fs::IStorage* pNodeFileStorage,
           fs::IStorage* pEntryFileStorage,
           fs::IStorage* pDataFileStorage,
           int entryCount,
           int64_t startOffset,
           int64_t endOffset,
           TFunc&& outputFunction
       ) NN_NOEXCEPT
{
    static const int TextLength = 128;
    static const int64_t StorageSizeMax = 0x80000000;

    // フラグメント数書き出し
    int dataOffset;
    {
        char text[TextLength];
        dataOffset = util::SNPrintf(text, TextLength, "Fragment count : %d\n", entryCount);

        NN_RESULT_DO(pDataFileStorage->Write(0, text, dataOffset));
    }

    fs::SubStorage storages[BucketTreeAnalyzer::StorageIndex_Max] =
    {
        fs::SubStorage(pTableFileStorage, 0, StorageSizeMax),
        fs::SubStorage(pNodeFileStorage, 0, StorageSizeMax),
        fs::SubStorage(pEntryFileStorage, 0, StorageSizeMax),
        fs::SubStorage(pDataFileStorage, dataOffset, StorageSizeMax),
    };

    BucketTreeAnalyzer analyzer;
    NN_RESULT_DO(analyzer.Initialize(
        storages,
        IndirectStorage::NodeSize,
        sizeof(IndirectStorage::Entry),
        entryCount,
        startOffset,
        "     VirtualOffset/    PhysicalOffset/StorageIndex",
        [](char* outText, size_t outSize, const void* pData) NN_NOEXCEPT -> size_t
        {
            const auto& entry = *reinterpret_cast<const IndirectStorage::Entry*>(pData);
            const size_t length = util::SNPrintf(
                outText, outSize, "0x%016llX 0x%016llX %d\n",
                entry.GetVirtualOffset(), entry.GetPhysicalOffset(), entry.storageIndex);
            return std::min(length, outSize - 1);
        },
        [](char* outText, size_t outSize, const void* pData) NN_NOEXCEPT -> size_t
        {
            const auto& entry = *reinterpret_cast<const IndirectStorage::Entry*>(pData);
            const size_t length = util::SNPrintf(
                outText, outSize, "%lld,%d\n",
                entry.GetVirtualOffset(), entry.storageIndex);
            return std::min(length, outSize - 1);
        }
    ));

    NN_RESULT_DO(outputFunction(analyzer));

    int64_t offset = 0;
    NN_RESULT_DO(analyzer.GetEntryStorage()->GetSize(&offset));
    offset += dataOffset;

    const auto storageSize = endOffset - startOffset;

    // 終端情報書き出し
    char text[TextLength];
    auto length = util::SNPrintf(text, TextLength, "%lld,-1\n", storageSize);
    return pDataFileStorage->Write(offset, text, length);
}

}}}
