﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <nn/fs/fs_Directory.h>
#include <nn/fssystem/fs_BucketTreeBuilder.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace fssystem { namespace utilTool {

/**
 * @brief   BucketTree の解析クラスです。
 */
class BucketTreeAnalyzer
{
public:
    // ストレージのインデックスです。
    enum StorageIndex
    {
        StorageIndex_TableFile,
        StorageIndex_NodeFile,
        StorageIndex_EntryFile,
        StorageIndex_DataFile,
        StorageIndex_Max
    };

    // データをフォーマットする関数です。
    typedef std::function<size_t (char* outText, size_t outSize, const void* ptr)> FormatFunction;

    // 出力テキストをフォーマットするクラスです。
    class TextFormatter
    {
    public:
        TextFormatter(char* buffer, size_t size) NN_NOEXCEPT
            : m_Buffer(buffer)
            , m_Capacity(size)
            , m_Length(0)
        {
            NN_SDK_REQUIRES_NOT_NULL(buffer);
        }

        bool Append(const char* format, ...) NN_NOEXCEPT
        {
            va_list list;
            va_start(list, format);
            m_Length += util::VSNPrintf(m_Buffer + m_Length, m_Capacity - m_Length, format, list);
            if( m_Capacity <= m_Length + 2 )
            {
                m_Buffer[m_Capacity - 2] = '\n';
                m_Buffer[m_Capacity - 1] = '\0';

                m_Length = m_Capacity - 1;
            }
            va_end(list);
            return m_Length < m_Capacity - 1;
        }

        void Append(char value) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_LESS(m_Length, m_Capacity - 1);
            m_Buffer[m_Length] = value;
            ++m_Length;
            m_Buffer[m_Length] = '\0';
        }

        void Back(size_t count) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_LESS_EQUAL(count, m_Length);
            m_Length -= count;
            m_Buffer[m_Length] = '\0';
        }

        size_t End() NN_NOEXCEPT
        {
            Append('\n');
            return m_Length;
        }

        size_t GetLength() const NN_NOEXCEPT
        {
            return m_Length;
        }

    private:
        char* const m_Buffer;
        const size_t m_Capacity;
        size_t m_Length;
    };

public:
    // コンストラクタです。
    BucketTreeAnalyzer() NN_NOEXCEPT;

    // 初期化します。
    Result Initialize(
               fs::SubStorage* pStorageArray,
               int storageCount,
               size_t nodeSize,
               size_t entrySize,
               int entryCount,
               int64_t startOffset,
               const char* const pEntryHeader,
               FormatFunction formatEntry,
               FormatFunction formatData
           ) NN_NOEXCEPT;

    // 初期化します。
    template< size_t N >
    Result Initialize(
               fs::SubStorage (&storageArray)[N],
               size_t nodeSize,
               size_t entrySize,
               int entryCount,
               int64_t startOffset,
               const char* const pEntryHeader,
               FormatFunction formatEntry,
               FormatFunction formatData
           ) NN_NOEXCEPT
    {
        return Initialize(
                   storageArray,
                   N,
                   nodeSize,
                   entrySize,
                   entryCount,
                   startOffset,
                   pEntryHeader,
                   formatEntry,
                   formatData
               );
    }

    // ノードストレージを取得します。
    fs::IStorage* GetNodeStorage() const NN_NOEXCEPT
    {
        return &m_NodeStorage;
    }

    // エントリストレージを取得します。
    fs::IStorage* GetEntryStorage() const NN_NOEXCEPT
    {
        return &m_EntryStorage;
    }

private:
    // ノードストレージです。
    class NodeStorage : public fs::IStorage, public fs::detail::Newable
    {
    public:
        NodeStorage() NN_NOEXCEPT
            : m_TableStorage()
            , m_NodeStorage()
            , m_pSeparator(nullptr)
            , m_SeparatorLength(0)
            , m_NodeSize(0)
            , m_StartOffset(0)
            , m_Position(0)
        {
        }

        virtual ~NodeStorage() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        void Initialize(
                 const fs::SubStorage& tableStorage,
                 const fs::SubStorage& nodeStorage,
                 size_t nodeSize,
                 int64_t startOffset,
                 const char* pSeparator
             ) NN_NOEXCEPT
        {
            m_TableStorage = tableStorage;
            m_NodeStorage = nodeStorage;
            m_pSeparator = pSeparator;
            m_SeparatorLength = strnlen(pSeparator, fs::EntryNameLengthMax);
            m_NodeSize = nodeSize;
            m_StartOffset = startOffset;
        }

        virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(offset);
            NN_UNUSED(buffer);
            NN_UNUSED(size);
            return fs::ResultUnsupportedOperation();
        }

        virtual Result Write(
                           int64_t offset,
                           const void* buffer,
                           size_t size
                       ) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result GetSize(int64_t* pOutSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(pOutSize);
            return fs::ResultUnsupportedOperation();
        }

    private:
        fs::SubStorage m_TableStorage;
        fs::SubStorage m_NodeStorage;
        const char* m_pSeparator;
        size_t m_SeparatorLength;
        size_t m_NodeSize;
        int64_t m_StartOffset;
        int64_t m_Position;
    };

    // エントリストレージです。
    class EntryStorage : public fs::IStorage, public fs::detail::Newable
    {
    public:
        EntryStorage() NN_NOEXCEPT
            : m_EntryStorage()
            , m_DataStorage()
            , m_FormatEntry(nullptr)
            , m_FormatData(nullptr)
            , m_TextBuffer(nullptr)
            , m_TextBufferSize(0)
            , m_NodeSize(0)
            , m_EntrySize(0)
            , m_StartOffset(0)
            , m_EntryPos(0)
            , m_DataPos(0)
            , m_IsOutputData(false)
        {
        }

        virtual ~EntryStorage() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        void Initialize(
                 const fs::SubStorage& entryStorage,
                 const fs::SubStorage& dataStorage,
                 char* textBuffer,
                 size_t textBufferSize,
                 size_t nodeSize,
                 size_t entrySize,
                 int64_t startOffset,
                 FormatFunction formatEntry,
                 FormatFunction formatData,
                 bool isOutputData
             ) NN_NOEXCEPT
        {
            m_EntryStorage = entryStorage;
            m_DataStorage = dataStorage;
            m_FormatEntry = formatEntry;
            m_FormatData = formatData;
            m_TextBuffer = textBuffer;
            m_TextBufferSize = textBufferSize;
            m_NodeSize = nodeSize;
            m_EntrySize = entrySize;
            m_StartOffset = startOffset;
            m_IsOutputData = isOutputData;
        }

        virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(offset);
            NN_UNUSED(buffer);
            NN_UNUSED(size);
            return fs::ResultUnsupportedOperation();
        }

        virtual Result Write(
                           int64_t offset,
                           const void* buffer,
                           size_t size
                       ) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result GetSize(int64_t* pOutSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutSize);
            *pOutSize = m_DataPos;
            NN_RESULT_SUCCESS;
        }

    private:
        fs::SubStorage m_EntryStorage;
        fs::SubStorage m_DataStorage;
        FormatFunction m_FormatEntry;
        FormatFunction m_FormatData;
        char* m_TextBuffer;
        size_t m_TextBufferSize;
        size_t m_NodeSize;
        size_t m_EntrySize;
        int64_t m_StartOffset;
        int64_t m_EntryPos;
        int64_t m_DataPos;
        bool m_IsOutputData;
    };

private:
    static const size_t TextLength = 256;

private:
    mutable NodeStorage m_NodeStorage;
    mutable EntryStorage m_EntryStorage;
    std::unique_ptr<char[]> m_TextBuffer;
    char m_Separator[TextLength + 1];
};

}}}
