﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdio>
#include <cstdlib>

#include <nn/nn_Common.h>

#include <nn/lmem/lmem_Common.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fssystem/dbm/fs_AllocationTable.h>
#include <nn/fssystem/dbm/fs_AllocationTableStorage.h>
#include <nn/fssystem/dbm/fs_BufferedAllocationTableStorage.h>
#include <nn/fssystem/dbm/fs_DbmUtils.h>
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.h>
#include <nn/fssystem/dbm/fs_FileSystemTemplate.h>
#include <nn/fssystem/dbm/fs_FileSystemControlArea.h>

namespace nn { namespace fssystem { namespace save {

typedef dbm::FileSystemControlArea FileSystemControlArea;
typedef dbm::AllocationTable AllocationTable;
typedef dbm::AllocationTableStorage AllocationTableStorage;
typedef dbm::BufferedAllocationTableStorage BufferedAllocationTableStorage;

typedef dbm::FileSystemObjectTemplate<dbm::DirectoryName,dbm::FileName> FileSystemObject;
typedef dbm::FileObjectTemplate<dbm::DirectoryName,dbm::FileName> FileObject;
typedef dbm::DirectoryObjectTemplate<dbm::DirectoryName,dbm::FileName> DirectoryObject;
typedef FileSystemObject::StorageIndex StorageIndex;

class ScopedFSLock
{
public:
    NN_IMPLICIT ScopedFSLock(os::Mutex* pLocker)
    : m_pLocker(pLocker)
    {
        if( m_pLocker != nullptr )
        {
            m_pLocker->Lock();
        }
    }

    ~ScopedFSLock()
    {
        if( m_pLocker != nullptr )
        {
            m_pLocker->Unlock();
        }
    }

private:
    os::Mutex* m_pLocker;
};

template <typename T>
class ScopedFinalizeObject
{
public:
    ScopedFinalizeObject() NN_NOEXCEPT
    {
    }

    ~ScopedFinalizeObject() NN_NOEXCEPT
    {
        m_Obj.Finalize();
    }

    T* operator->() NN_NOEXCEPT
    {
        return &m_Obj;
    }

    T& Get() NN_NOEXCEPT
    {
        return m_Obj;
    }

    T* Ptr() NN_NOEXCEPT
    {
        return &m_Obj;
    }

private:
    T m_Obj;
};

template <typename T>
class ScopedFinalizer
{
public:
    NN_IMPLICIT ScopedFinalizer(T* obj) NN_NOEXCEPT
    : m_pObj(obj)
    {
    }

    ~ScopedFinalizer() NN_NOEXCEPT
    {
        if( m_pObj != nullptr )
        {
            m_pObj->Finalize();
        }
    }

    void Release() NN_NOEXCEPT
    {
        m_pObj = nullptr;
    }

private:
    T* m_pObj;
};

template <typename T>
class ScopedCloser
{
public:
    NN_IMPLICIT ScopedCloser(T* obj) NN_NOEXCEPT
    : m_pObj(obj)
    {
    }

    ~ScopedCloser() NN_NOEXCEPT
    {
        if( m_pObj != nullptr )
        {
            m_pObj->Close();
        }
    }

    void Release() NN_NOEXCEPT
    {
        m_pObj = nullptr;
    }

private:
    T* m_pObj;
};

inline bool IsPower2(int x) NN_NOEXCEPT
{
    return (0 == (x & (x - 1)));
}

inline uint32_t ILog2(uint32_t x) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(x, 0U);
    return (31 - dbm::CountLeadingZeros(x));
}

inline uint32_t CeilPower2(uint32_t x) NN_NOEXCEPT
{
    if(x == 0U)
    {
        return 1U;
    }
    return (0x80000000 >> (dbm::CountLeadingZeros(x - 1) - 1));
}

}}}

