﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fssystem/dbm/fs_DbmParameters.h>
#include <nn/fssystem/save/fs_SaveTypes.h>

namespace nn { namespace fssystem { namespace save {

class Path : public fs::detail::Newable
{
public:
    typedef dbm::PathChar Char;

private:
    Char m_Buf[nn::fs::EntryNameLengthMax + 1];

public:
    Path() NN_NOEXCEPT
    {
        m_Buf[0] = '\0';
    }

    Path(const Path& rhs) NN_NOEXCEPT
    {
        memcpy(m_Buf, rhs.m_Buf, sizeof(m_Buf));
    }

    nn::Result Initialize(const char* p) NN_NOEXCEPT
    {
        size_t len = strnlen(p, nn::fs::EntryNameLengthMax + 1);
        if( len > nn::fs::EntryNameLengthMax )
        {
            return nn::fs::ResultTooLongPath();
        }
        memcpy(m_Buf, p, len);
        m_Buf[len] = '\0';
        NN_RESULT_SUCCESS;
    }

    Result InitializeNormalized(const char* path) NN_NOEXCEPT
    {
        size_t length = 0;
        return PathTool::Normalize(m_Buf, &length, path, sizeof(m_Buf));
    }

    Result SetChildPath(const char* childName) NN_NOEXCEPT;

    Result SetParentPath() NN_NOEXCEPT;

    size_t GetLength() const NN_NOEXCEPT
    {
        return strnlen(m_Buf, nn::fs::EntryNameLengthMax);
    }

    const Char* GetString() const NN_NOEXCEPT
    {
        return m_Buf;
    }

    bool IsEmptyPath() const NN_NOEXCEPT
    {
        return (m_Buf[0] == '\0');
    }

public:
    /**
    * @brief パスが等しいかどうか比較します。
    *
    * @param[in] str1 パス
    * @param[in] str2 パス
    *
    * @return パスが等しければ true、そうでなければ false
    */
    static bool IsEqualPath(const Path& str1, const Path& str2) NN_NOEXCEPT;

    /**
    * @brief パスが別のパスの部分文字列かどうか判定します。
    *
    * @param[in] strFullset パス
    * @param[in] strSubset 含まれているかを調べるパス
    *
    * @return strSubset が strFullset に含まれていれば true、そうでばければ false
    */
    static bool IsContainingPath(const Path& strFullset, const Path& strSubset) NN_NOEXCEPT;

    /**
    * @brief ディレクトリと、その中に入っているファイルやディレクトリという関係かどうかを判定します。
    *        (/dir1 と　/dir1/file という関係)
    *
    * @param[in] strDirectory ディレクトリパス
    * @param[in] strSubEntry サブエントリーパス
    *
    * @return strSubEntry が strDirectory の子ファイル、ディレクトリであれば true、そうでなければ false
    */
    static bool IsInsidePath(const Path& strDirectory, const Path& strSubEntry) NN_NOEXCEPT;
};

}}}

