﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/fssystem/save/fs_ISaveFile.h>

namespace nn { namespace fssystem { namespace save {

/**
* @brief 下位ファイルオブジェクトに対して透過的に操作を行うファイルオブジェクトです。
*
*        複数のアーカイブを跨いで異なる IFile を扱う場合に使用します。
*        SubFileSetと近い関係ですが、こちらは
*
*        ・オフセットやサイズは書き換えません。
*        ・ファイルのリサイズ、自動拡張に対応します。
*/
class FileInFile : public IFile
{
    NN_DISALLOW_COPY(FileInFile);

public:
    /**
    * @brief        コンストラクタ
    *
    * @param[in]    pFile   下位ファイル
    */
    explicit FileInFile(IFile* pFile) NN_NOEXCEPT
    : m_pFile(pFile)
    {
    }

    /**
    * @brief        コンストラクタ
    */
    FileInFile() NN_NOEXCEPT
    : m_pFile(nullptr)
    {
    }

    /**
    * @brief        ファイルオブジェクトへの参照を設定します。
    *
    * @param[in]    pFile   下位ファイルオブジェクト
    */
    void Initialize(IFile* pFile) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pFile == nullptr);
        NN_SDK_ASSERT_NOT_NULL(pFile);
        m_pFile = pFile;
    }

    /**
    * @brief        下位ファイルオブジェクトへの参照を解除します。
    */
    void Finalize() NN_NOEXCEPT
    {
        m_pFile = nullptr;
    }

    /**
    * @brief        下位ファイルを取得します。
    *
    * @return       下位ファイルへのポインタ
    */
    IFile* GetFile() NN_NOEXCEPT
    {
        return m_pFile;
    }

    /**
    * @brief        ファイルの内容をバッファに読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result ReadBytes(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        return m_pFile->ReadBytes(offset, buffer, size);
    }

    /**
    * @brief        バッファの内容をファイルに書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    * @param[in]    option  書き込みオプション
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result WriteBytes(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        return m_pFile->WriteBytes(offset, buffer, size);
    }

    /**
    * @brief        ファイルサイズを取得します。
    *
    * @param[out]   outValue    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        return m_pFile->GetSize(outValue);
    }

    /**
    * @brief        ファイルサイズを変更します。
    *
    * @param[in]    size    変更後のファイルサイズ
    *
    * @return       失敗の結果を返します。
    */
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        return m_pFile->SetSize(size);
    }

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        return m_pFile->Flush();
    }

    /**
    * @brief        範囲指定処理を行います。
    *
    * @param[out]   outBuffer       範囲指定処理の結果を格納するバッファ
    * @param[in]    outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]    operationId     範囲指定処理の種類
    * @param[in]    offset          範囲指定処理開始位置
    * @param[in]    size            範囲指定処理を行うデータサイズ
    * @param[in]    inBuffer        範囲指定処理に渡すバッファ
    * @param[in]    inBufferSize    範囲指定処理に渡すバッファのサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        return m_pFile->OperateRange(
            outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
    }

private:
    IFile* m_pFile;     //!< 下位ファイル
};

}}}

