﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdio>
#include <cstdlib>

#include <nn/nn_Common.h>

#include <nn/fssystem/save/fs_SaveTypes.h>
#include <nn/fssystem/save/fs_SavePath.h>
#include <nn/fssystem/save/fs_ISaveFile.h>
#include <nn/fssystem/save/fs_ISaveDirectory.h>

namespace nn { namespace fssystem { namespace save {

class IFileSystem
{
    NN_DISALLOW_COPY(IFileSystem);

public:
    /**
    * @brief        コンストラクタです。
    */
    IFileSystem() NN_NOEXCEPT
    {
    }

    /**
    * @brief        デストラクタです。
    */
    virtual ~IFileSystem() NN_NOEXCEPT
    {
    }

    /**
    * @brief        ファイルを作成します。
    *
    * @param[in]    path    ファイルパス
    * @param[in]    size    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateFile(const Path& path, int64_t size) NN_NOEXCEPT = 0;

    /**
    * @brief        ディレクトリを作成します。
    *
    * @param[in]    path ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateDirectory(const Path& path) NN_NOEXCEPT = 0;

    /**
    * @brief        ファイルを開きます。
    *
    * @param[out]   outValue    ファイルオブジェクト
    * @param[in]    path        ファイルパス
    * @param[in]    mode        オープンモード
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OpenFile(IFile** outValue, const Path& path, int mode) NN_NOEXCEPT = 0;

    /**
    * @brief        ディレクトリを開きます。
    *
    * @param[out]   outValue    ディレクトリオブジェクト
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OpenDirectory(IDirectory** outValue, const Path& path) NN_NOEXCEPT = 0;

    /**
    * @brief        ファイルを閉じます。
    *
    * @param[in]    pFile   ファイルオブジェクト
    */
    virtual void CloseFile(IFile* pFile) NN_NOEXCEPT = 0;

    /**
    * @brief        ディレクトリを閉じます。
    *
    * @param[in]    pDirectory  ディレクトリオブジェクト
    */
    virtual void CloseDirectory(IDirectory* pDirectory) NN_NOEXCEPT = 0;

    /**
    * @brief        ファイルを削除します。
    *
    * @param[in]    path    ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteFile(const Path& path) NN_NOEXCEPT = 0;

    /**
    * @brief        ディレクトリを削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteDirectory(const Path& path) NN_NOEXCEPT = 0;

    /**
    * @brief        ディレクトリを再帰的に削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteDirectoryRecursively(const Path& path) NN_NOEXCEPT
    {
        return DeleteDirectoryRecursivelyInternal(path, true);
    }

    /**
    * @brief        全ての子エントリを再帰的に削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CleanDirectoryRecursively(const Path& path) NN_NOEXCEPT
    {
        return DeleteDirectoryRecursivelyInternal(path, false);
    }

    /**
    * @brief        ファイルをリネームします。
    *
    * @param[in]    oldPath 変更対象のファイルパス
    * @param[in]    newPath 変更後のファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameFile(const Path& oldPath, const Path& newPath) NN_NOEXCEPT = 0;

    /**
    * @brief        ディレクトリをリネームします。
    *
    * @param[in]    oldPath 変更対象のディレクトリパス
    * @param[in]    newPath 変更後のディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameDirectory(const Path& oldPath, const Path& newPath) NN_NOEXCEPT = 0;

    /**
    * @brief        指定したファイルが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ファイルが存在しているかどうか
    * @param[in]    path        ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasFile(bool* outValue, const Path& path) NN_NOEXCEPT;

    /**
    * @brief        指定したディレクトリが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ディレクトリが存在しているかどうか
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasDirectory(bool* outValue, const Path& path) NN_NOEXCEPT;

    /**
    * @brief        空き領域のサイズをバイト数単位で取得します。
    *
    * @param[out]   outValue    空き領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetFreeBytes(int64_t*) NN_NOEXCEPT
    {
        return nn::fs::ResultUnsupportedOperation();
    }

    /**
    * @brief        実データ領域のサイズをバイト数単位で取得します。
    *
    * @param[out]   outValue    実データ領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetDataAreaBytes(int64_t*) NN_NOEXCEPT
    {
        return nn::fs::ResultUnsupportedOperation();
    }

private:
    class ScopedDirectoryAccessor
    {
    private:
        IFileSystem* m_pArchive;
        IDirectory* m_pDir;

    public:
        explicit ScopedDirectoryAccessor(IFileSystem* pArchive) NN_NOEXCEPT
            : m_pArchive(pArchive)
            , m_pDir(nullptr)
        {
        }

        ~ScopedDirectoryAccessor() NN_NOEXCEPT
        {
            if( m_pDir != nullptr )
            {
                m_pArchive->CloseDirectory(m_pDir);
                m_pDir = nullptr;
            }
        }

        Result Open(const Path& path) NN_NOEXCEPT
        {
            IDirectory* pDirectory;
            NN_RESULT_DO(m_pArchive->OpenDirectory(&pDirectory, path));
            m_pDir = pDirectory;
            NN_RESULT_SUCCESS;
        }

        IDirectory* operator->() const NN_NOEXCEPT { return m_pDir; }
    };

private:
    Result DeleteDirectoryRecursivelyInternal(
               const Path& path,
               bool deleteItself
           ) NN_NOEXCEPT;

    Result DeleteDirectoryRecursivelyInternal(
               Path* pPath,
               nn::fs::DirectoryEntry* pEntry,
               bool deleteItself
           ) NN_NOEXCEPT;
};

}}}
