﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fssystem/save/fs_SaveTypes.h>

namespace nn { namespace fssystem { namespace save {

class IFile
{
    NN_DISALLOW_COPY(IFile);

public:
    /**
    * @brief        コンストラクタです。
    */
    IFile() NN_NOEXCEPT
    {
    }

    /**
    * @brief        デストラクタです。
    */
    virtual ~IFile() NN_NOEXCEPT
    {
    }

    /**
    * @brief        ファイルの内容をバッファに読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result ReadBytes(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT = 0;

    /**
    * @brief        バッファの内容をファイルに書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result WriteBytes(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT = 0;

    /**
    * @brief        ファイルサイズを取得します。
    *
    * @param[out]   outValue    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT = 0;

    /**
    * @brief        ファイルサイズを変更します。
    *
    * @param[in]    size    変更後のファイルサイズ
    *
    * @return       失敗の結果を返します。
    */
    virtual Result SetSize(int64_t size) NN_NOEXCEPT = 0;

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT = 0;

    /**
    * @brief        範囲指定処理を行います。
    *
    * @param[out]   outBuffer       範囲指定処理の結果を格納するバッファ
    * @param[in]    outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]    operationId     範囲指定処理の種類
    * @param[in]    offset          範囲指定処理開始位置
    * @param[in]    size            範囲指定処理を行うデータサイズ
    * @param[in]    inBuffer        範囲指定処理に渡すバッファ
    * @param[in]    inBufferSize    範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(false);
        NN_UNUSED(outBuffer);
        NN_UNUSED(outBufferSize);
        NN_UNUSED(operationId);
        NN_UNUSED(offset);
        NN_UNUSED(size);
        NN_UNUSED(inBuffer);
        NN_UNUSED(inBufferSize);
        return nn::fs::ResultUnsupportedOperation();
    }
};

}}}

