﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>

#include <nn/nn_Common.h>

#include <nn/fssystem/dbm/fs_DuplexBitmap.h>
#include <nn/fssystem/save/fs_SaveFileStorage.h>

class FsDuplexBitmapHolderTest;

namespace nn { namespace fssystem { namespace save {

/**
* @brief 二重化選択ビットマップ管理レイヤーです。
*/
class DuplexBitmapHolder : public dbm::DuplexBitmap
{
    NN_DISALLOW_COPY(DuplexBitmapHolder);

public:
    /**
    * @brief        コンストラクタ
    */
    DuplexBitmapHolder() NN_NOEXCEPT;

    /**
    * @brief        二重化選択ビットマップをフォーマットします。
    *
    * @param[in]    blockCount          総ブロック数
    * @param[in]    storage             メタデータ用ストレージ
    * @param[in]    storageOriginal     オリジナルメタデータ用ストレージ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Format(
                      uint32_t blockCount,
                      fs::SubStorage storage,
                      fs::SubStorage storageOriginal
                  ) NN_NOEXCEPT;

    /**
    * @brief        二重化選択ビットマップを拡張します。
    *
    * @param[in]    blockCountOld       拡張前の総ブロック数
    * @param[in]    blockCountNew       拡張後の総ブロック数
    * @param[in]    storage             メタデータ用ストレージ
    * @param[in]    storageOriginal     オリジナルメタデータ用ストレージ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Expand(
                      uint32_t blockCountOld,
                      uint32_t blockCountNew,
                      fs::SubStorage storage,
                      fs::SubStorage storageOriginal
                  ) NN_NOEXCEPT;

    /**
    * @brief        二重化選択ビットマップをマウントします。
    *
    * @param[in]    blockCount          総ブロック数
    * @param[in]    storage             メタデータ用ストレージ
    * @param[in]    storageOriginal     オリジナルメタデータ用ストレージ
    *
    * @return       関数の処理結果を返します。
    */
    void Initialize(
             uint32_t blockCount,
             fs::SubStorage storage,
             fs::SubStorage storageOriginal
         ) NN_NOEXCEPT;

    /**
    * @brief        二重化選択ビットマップを読み込み扱いでマウントします。
    *
    * @param[in]    blockCount          総ブロック数
    * @param[in]    storage             メタデータ用ストレージ
    * @param[in]    storageOriginal     オリジナルメタデータ用ストレージ
    *
    * @return       関数の処理結果を返します。
    */
    void InitializeForRead(
             uint32_t blockCount,
             fs::SubStorage storage,
             fs::SubStorage storageOriginal
         ) NN_NOEXCEPT;

    /**
    * @brief        二重化選択ビットマップを書き込み扱いで再マウントします。
    */
    void RemountForWrite() NN_NOEXCEPT
    {
        DuplexBitmap::Finalize();
        DuplexBitmap::Initialize(
            m_CountBlock,
            m_Storage,
            m_StorageOriginal
        );
    }

    /**
    * @brief        二重化選択ビットマップを持つストレージを取得します。(オリジナル系)
    *
    * @return       設定対象ストレージ
    */
    const fs::SubStorage& GetOriginalStorage() NN_NOEXCEPT
    {
        return m_StorageOriginal;
    }

    /**
    * @brief        二重化選択ビットマップを持つストレージを取得します。(更新系)
    *
    * @return       設定対象ストレージ
    */
    const fs::SubStorage& GetUpdateStorage() NN_NOEXCEPT
    {
        return m_Storage;
    }

    /**
    * @brief        ブロックカウントを取得します。
    *
    * @return       ブロックカウントを返します。
    */
    uint32_t GetBlockCount() const NN_NOEXCEPT
    {
        return m_CountBlock;
    }

private:
    /**
    * @brief        二重化選択ビットマップの内部データを交換します。
    *
    * @param[out]   pOutBitmap  "更新系"のビットマップ
    *
    * @details      これは更新系ビットマップと、参照系ビットマップを入れ替えする際に使用します。
    *               HierarchicalDuplexFile の実装に強く依存するため、他からこの関数を呼ぶべきではありません。
    */
    void SwapDuplexBitmapForHierarchicalDuplexStorage(DuplexBitmapHolder* pOutBitmap) NN_NOEXCEPT;
    friend class HierarchicalDuplexStorage;

private:
    fs::SubStorage m_Storage;
    fs::SubStorage m_StorageOriginal;
    uint32_t m_CountBlock;

private:
    // テスト用クラスには内部データを公開します。
    friend class ::FsDuplexBitmapHolderTest;
};

}}}

