﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/buffers/fs_IBufferManager.h>

namespace nn { namespace fssystem { namespace save {

//! キャッシュ付きストレージクラス
class BufferedStorage : public fs::IStorage
{
    NN_DISALLOW_COPY(BufferedStorage);

public:
    /**
    * @brief      コンストラクタです。
    */
    BufferedStorage() NN_NOEXCEPT;

    /**
    * @brief      デストラクタです。
    */
    virtual ~BufferedStorage() NN_NOEXCEPT;

    /**
    * @brief      キャッシュが有効なストレージとしてマウントします。
    *
    * @param[in]  baseStorage       キャッシュ対象ストレージ
    * @param[in]  pBufferManager    バッファマネージャ
    * @param[in]  blockSize         キャッシュするブロックのサイズ
    * @param[in]  bufferCount       キャッシュバッファ数
    *
    * @return     関数の処理結果を返します。
    *
    * @pre
    *             - pBaseStorage != nullptr
    *             - pBufferManager != nullptr
    *             - 0 < blockSize
    *             - blockSize が 2 のべき乗
    *             - 0 < bufferCount
    */
    Result Initialize(
               fs::SubStorage baseStorage,
               IBufferManager* pBufferManager,
               size_t blockSize,
               int bufferCount
           ) NN_NOEXCEPT;

    /**
    * @brief      アンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief      初期化済みかどうか取得します。
    *
    * @return     初期化済みかどうかを返します。
    */
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_pCaches != nullptr;
    }

    /**
    * @brief      ストレージの内容をバッファに読み込みます。
    *
    * @param[in]  offset  読み込み開始位置
    * @param[out] buffer  読み込んだ内容をコピーするバッファ
    * @param[in]  size    読み込むデータサイズ
    *
    * @return     関数の処理結果を返します。
    *
    * @pre
    *             - Initialize に成功している
    */
    virtual Result Read(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief      バッファの内容をストレージに書き込みます。
    *
    * @param[in]  offset  書き込み開始位置
    * @param[in]  buffer  書き込むデータ
    * @param[in]  size    書き込むデータサイズ
    *
    * @return     関数の処理結果を返します。
    *
    * @pre
    *             - Initialize に成功している
    */
    virtual Result Write(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ストレージサイズを取得します。
    *
    * @param[out]   outValue    ストレージサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ストレージサイズを変更します。
    *
    * @param[in]    size    変更後のストレージサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    using IStorage::OperateRange;

    /**
    * @brief        キャッシュを無効化します。
    */
    void InvalidateCaches() NN_NOEXCEPT;

    /**
    * @brief        バッファマネージャを取得します。
    *
    * @return       バッファマネージャを返します。
    */
    IBufferManager* GetBufferManager() const NN_NOEXCEPT
    {
        return m_pBufferManager;
    }

    /**
    * @brief        一括読み込みを有効にします。
    */
    void EnableBulkRead() NN_NOEXCEPT
    {
        m_IsBulkReadEnabled = true;
    }

private:
    class Cache;
    class UniqueCache;
    class SharedCache;

private:
    /**
    * @brief        バッファを新規確保する前の準備を行います。
    *
    * @return       関数の処理結果を返します。
    */
    Result PrepareAllocation() NN_NOEXCEPT;

    /**
    * @brief        ダーティキャッシュが多くならないように適宜フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    Result ControlDirtiness() NN_NOEXCEPT;

    /**
    * @brief        ストレージの内容をバッファに読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    Result ReadCore(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;

    /**
    * @brief        読み込み領域の先頭をキャッシュから読み込み、読み込み領域を狭めます。
    *
    * @param[in,out]    pOffset         読み込み開始位置
    * @param[in]        buffer          読み込んだ内容をコピーするバッファ
    * @param[in,out]    pSize           読み込むデータサイズ
    * @param[in,out]    pBufferOffset   読み込んだ内容をコピーするバッファのオフセット
    *
    * @return       読み込み領域の先頭をキャッシュする必要があれば true を返します。
    */
    bool ReadHeadCache(int64_t* pOffset, void* buffer, size_t* pSize, int64_t* pBufferOffset) NN_NOEXCEPT;

    /**
    * @brief        読み込み領域の末尾をキャッシュから読み込み、読み込み領域を狭めます。
    *
    * @param[in]        offset          読み込み開始位置
    * @param[in]        buffer          読み込んだ内容をコピーするバッファ
    * @param[in,out]    pSize           読み込むデータサイズ
    * @param[in]        bufferOffset    読み込んだ内容をコピーするバッファのオフセット
    *
    * @return       読み込み領域の末尾をキャッシュする必要があれば true を返します。
    */
    bool ReadTailCache(int64_t offset, void* buffer, size_t* pSize, int64_t bufferOffset) NN_NOEXCEPT;

    /**
    * @brief        ストレージの内容をバッファに一括で読み込みます。
    *
    * @param[in]    offset              読み込み開始位置
    * @param[out]   buffer              読み込んだ内容をコピーするバッファ
    * @param[in]    size                読み込むデータサイズ
    * @param[in]    isHeadCacheNeeded   読み込み領域の先頭をキャッシュする必要があるかどうか
    * @param[in]    isTailCacheNeeded   読み込み領域の末尾をキャッシュする必要があるかどうか
    *
    * @return       関数の処理結果を返します。
    */
    Result BulkRead(int64_t offset, void* buffer, size_t size, bool isHeadCacheNeeded, bool isTailCacheNeeded) NN_NOEXCEPT;

    /**
    * @brief      バッファの内容をストレージに書き込みます。
    *
    * @param[in]  offset  書き込み開始位置
    * @param[in]  buffer  書き込むデータ
    * @param[in]  size    書き込むデータサイズ
    *
    * @return     関数の処理結果を返します。
    *
    * @pre
    *             - Initialize に成功している
    *             - buffer != nullptr
    */
    Result WriteCore(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;

private:
    fs::SubStorage m_BaseStorage;       //!< ベースストレージ
    IBufferManager* m_pBufferManager;   //!< バッファマネージャ
    size_t m_BlockSize;                 //!< ブロックサイズ
    int64_t m_BaseStorageSize;          //!< ベースストレージのサイズ
    std::unique_ptr<Cache[]> m_pCaches; //!< キャッシュ配列
    int m_CacheCount;                   //!< キャッシュ数
    Cache* m_pNextAcquireCache;         //!< 次に獲得するキャッシュ
    Cache* m_pNextFetchCache;           //!< 次にフェッチするキャッシュ
    nn::os::Mutex m_Mutex;              //!< ミューテックス
    bool m_IsBulkReadEnabled;           //!< 一括読み込みが有効かどうか
};

}}}

