﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fsa/fs_IFile.h>

namespace nn { namespace fssystem {

class ZeroBitmapFile : public fs::fsa::IFile, public fs::detail::Newable
{
    NN_DISALLOW_COPY(ZeroBitmapFile);

public:
    ZeroBitmapFile() NN_NOEXCEPT
        : m_pBaseFile(nullptr),
          m_pBitmap(nullptr),
          m_BlockSize(0),
          m_Mode(static_cast<nn::fs::OpenMode>(0))
    {
    }

    virtual ~ZeroBitmapFile() NN_NOEXCEPT NN_OVERRIDE {}

    Result Initialize(
        std::unique_ptr<fs::fsa::IFile> pBaseFile,
        std::unique_ptr<Bit8[], nn::fs::detail::Deleter> pBitmap,
        int64_t bitmapSize,
        int64_t blockSize,
        fs::OpenMode mode
    ) NN_NOEXCEPT;

private:
    static const auto BitsPerByte = 8;

private:
    virtual Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoWrite(int64_t offset, const void* buffer, size_t size, const fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoOperateRange(
        void* outBuffer,
        size_t outBufferSize,
        fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

    bool IsFree(int64_t offset) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
        NN_SDK_REQUIRES_LESS(offset / m_BlockSize / BitsPerByte, m_BitmapSize);
        return (m_pBitmap.get()[(offset / m_BlockSize) / BitsPerByte] & (1 << ((offset / m_BlockSize) % BitsPerByte))) != 0;
    }

    void Clear(int64_t offset) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
        NN_SDK_REQUIRES_LESS(offset / m_BlockSize / BitsPerByte, m_BitmapSize);
        m_pBitmap.get()[(offset / m_BlockSize) / BitsPerByte] &= ~(1 << ((offset / m_BlockSize) % BitsPerByte));
    }

private:
    std::unique_ptr<fs::fsa::IFile> m_pBaseFile;
    std::unique_ptr<Bit8[], nn::fs::detail::Deleter> m_pBitmap;
    int64_t m_BlockSize;
    int64_t m_BitmapSize;
    fs::OpenMode m_Mode;
};

}}
