﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/util/util_BinTypes.h>

namespace nn { namespace fssystem {

#pragma pack(push, 1)

    // TODO: crypto から定義を引っ張ってくる等
    struct Hash
    {
        static const int Size = 32; // TBD?
        uint8_t value[Size];
    };
    NN_STATIC_ASSERT(std::is_pod<Hash>::value);

    typedef Hash NcaDigest;

    struct NcaHeader
    {
        enum class ContentType : uint8_t
        {
            Program = 0, // ACID を持ち、検証するタイプ
            Meta,
            Control,
            Manual,
            Data,       // システムデータ
            PublicData, // アプリケーションが参照可能なシステムデータ及び追加コンテンツ
        };

        enum class DistributionType : uint8_t
        {
            Download = 0,
            GameCard
        };

        enum DecryptionKey
        {
            DecryptionKey_AesXts = 0,
            DecryptionKey_AesXts1 = 0,
            DecryptionKey_AesXts2,
            DecryptionKey_AesCtr,
            DecryptionKey_AesCtrEx,
            DecryptionKey_AesCtrHw,
            DecryptionKey_Count
        };

        enum class EncryptionType : uint8_t
        {
            Auto = 0,
            None,
        };

        struct FsInfo
        {
            uint32_t startSector;
            uint32_t endSector;
            uint32_t hashTargetSize;
            uint32_t reserve;
        };
        NN_STATIC_ASSERT(std::is_pod<FsInfo>::value);

        static const uint32_t Signature = NN_UTIL_CREATE_SIGNATURE_4('N', 'C' ,'A' ,'3');
        static const int Size = 1024;
        static const int FsCountMax = 4;
        static const int HeaderSignSize = 256;
        static const int HeaderSignCount = 2;
        static const int EncryptedKeyAreaSize = 256;
        static const int RightIdSize = 16;

        static const int Log2SectorSize = 9; // 512bytes. TBD
        static const int XtsBlockSize = 512;
        static const int CtrBlockSize = 16;

        uint8_t headerSign1[HeaderSignSize];
        uint8_t headerSign2[HeaderSignSize];

        uint32_t signature;
        DistributionType distributionType;
        ContentType contentType;
        uint8_t keyGeneration; // 鍵世代 2 まではこちらに格納、鍵世代が 3 以降は 0 もしくは 2
        uint8_t keyIndex;

        uint64_t contentSize;
        uint64_t programId;
        uint32_t contentIndex;
        uint32_t sdkAddonVersion;
        uint8_t keyGeneration2; // 鍵世代 3 以降はこちらに格納。鍵世代 2 までは 0。
        uint8_t reserve1[3];
        uint32_t reserve2[3];
        uint8_t  rightsId[RightIdSize];

        FsInfo fsInfo[FsCountMax];
        Hash   fsHeaderHash[FsCountMax];

        uint8_t encryptedKey[EncryptedKeyAreaSize];

    public:
        static uint64_t SectorToByte(uint32_t sector) NN_NOEXCEPT
        {
            return static_cast<uint64_t>(sector) << Log2SectorSize;
        }
        static uint32_t ByteToSector(uint64_t byte) NN_NOEXCEPT
        {
            return static_cast<uint32_t>(byte >> Log2SectorSize);
        }
        uint8_t GetProperKeyGeneration() const NN_NOEXCEPT;

        Result Verify() const NN_NOEXCEPT; // AuthoringTool 向け
    };

    NN_STATIC_ASSERT(sizeof(NcaHeader) == 1024);
    NN_STATIC_ASSERT(std::is_pod<NcaHeader>::value);

    struct NcaBucketInfo
    {
        static const int HeaderSize = 16;

        int64_t offset;          // テーブルデータまでのオフセット
        int64_t size;            // テーブルデータのサイズ
        char header[HeaderSize]; // テーブルのヘッダ情報
    };
    NN_STATIC_ASSERT(std::is_pod<NcaBucketInfo>::value);

    struct NcaPatchInfo
    {
        static const int Size = 64;
        static const int Offset = 0x100;

        int64_t indirectOffset;
        int64_t indirectSize;
        char indirectHeader[NcaBucketInfo::HeaderSize];
        int64_t aesCtrExOffset;
        int64_t aesCtrExSize;
        char aesCtrExHeader[NcaBucketInfo::HeaderSize];

        bool HasIndirectTable() const NN_NOEXCEPT;
        bool HasAesCtrExTable() const NN_NOEXCEPT;
    };
    NN_STATIC_ASSERT(std::is_pod<NcaPatchInfo>::value);

    union NcaAesCtrUpperIv
    {
        uint64_t value;
        struct Part
        {
            uint32_t generation;
            uint32_t secureValue;
        }
        part;
    };
    NN_STATIC_ASSERT(std::is_pod<NcaAesCtrUpperIv>::value);

    struct NcaSparseInfo
    {
        NcaBucketInfo bucket;   // SparseStorage のテーブル情報
        int64_t physicalOffset; // 実ストレージのオフセット
        uint16_t generation;    // SparseTable の世代番号
        char _reserved[6];

        int64_t GetPhysicalSize() const NN_NOEXCEPT
        {
            return this->bucket.offset + this->bucket.size;
        }

        uint32_t GetGeneration() const NN_NOEXCEPT
        {
            // 元データと世代番号が被る可能性を下げるためシフトする
            return uint32_t(this->generation) << 16;
        }

        const NcaAesCtrUpperIv MakeAesCtrUpperIv(NcaAesCtrUpperIv upperIv) const NN_NOEXCEPT
        {
            NcaAesCtrUpperIv sparseUpperIv = upperIv;
            sparseUpperIv.part.generation = GetGeneration();
            return sparseUpperIv;
        }
    };
    NN_STATIC_ASSERT(std::is_pod<NcaSparseInfo>::value);

    struct NcaFsHeader
    {
        static const int Size = 512; // TBD
        static const int HashDataOffset = 8;
        static const uint32_t ProgramIdOffsetMaskOnSecureValue = 0xFF000000;

        enum class FsType : uint8_t
        {
            RomFs = 0,
            PartitionFs,
        };

        enum class EncryptionType : uint8_t
        {
            Auto = 0,
            None,
            AesXts,
            AesCtr,
            AesCtrEx
        };

        enum class HashType : uint8_t
        {
            Auto = 0,
            None,
            HierarchicalSha256Hash,
            HierarchicalIntegrityHash,
        };

        struct Region
        {
            int64_t offset;
            int64_t size;
        };
        NN_STATIC_ASSERT(std::is_pod<Region>::value);

        union HashData
        {
            struct HierarchicalSha256Data
            {
                static const int HashLayerCountMax = 5;
                static const int MasterHashOffset;

            public:
                Hash fsDataMasterHash;
                int32_t hashBlockSize;
                int32_t hashLayerCount;
                Region hashLayerRegion[HashLayerCountMax];

            public:
                Result Verify() const NN_NOEXCEPT;

            } hierarchicalSha256Data;
            NN_STATIC_ASSERT(std::is_pod<HierarchicalSha256Data>::value);

            struct IntegrityMetaInfo
            {
                static const int MasterHashOffset;

            public:
                uint32_t magic;
                uint32_t version;
                uint32_t sizeMasterHash;

                struct InfoLevelHash
                {
                public:
                    uint32_t maxLayers;

                    struct HierarchicalIntegrityVerificationLevelInformation
                    {
                        static const int IntegrityMaxLayerCount = 7;
                    public:
                        int64_t offset;
                        int64_t size;
                        int32_t orderBlock;
                        char    reserved[4];
                    } info[HierarchicalIntegrityVerificationLevelInformation::IntegrityMaxLayerCount - 1];

                    struct SignatureSalt
                    {
                        static const auto Size = 32; //!< ソルトのサイズ

                    public:
                        char value[Size]; //!< ソルト
                    } seed;

                } infoLevelHash;

                Hash masterHash;

            public:
                Result Verify() const NN_NOEXCEPT;

            } integrityMetaInfo;
            NN_STATIC_ASSERT(std::is_pod<IntegrityMetaInfo>::value);

        public:
            uint8_t reserve[NcaPatchInfo::Offset - HashDataOffset];
        };

    public:
        uint16_t version;
        FsType   fsType;
        HashType hashType;
        EncryptionType encryptionType;
        char reserved[3];
        HashData hashData;
        NcaPatchInfo patchInfo;
        NcaAesCtrUpperIv aesCtrUpperIv;
        NcaSparseInfo sparseInfo;
        char reserved2[0x88];

    public:
        Result Verify() const NN_NOEXCEPT;
    };
    NN_STATIC_ASSERT(std::is_pod<NcaFsHeader>::value);

#pragma pack(pop)

}}
