﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Thread.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fssystem/fs_AsynchronousRunner.h>

namespace nn { namespace fssystem {

class ThreadPool;

//! 非同期アクセスレイヤ
class AsynchronousAccessFile : public fs::fsa::IFile, public fs::detail::Newable
{
    NN_DISALLOW_COPY(AsynchronousAccessFile);

public:
    /**
    * @brief      コンストラクタです。
    *
    * @param[in]  pBaseFile   ファイル
    * @param[in]  mode        ファイルへのアクセスモード（OpenMode 型の要素、またはそのビット和）
    * @param[in]  pThreadPool スレッドプール
    *
    * @pre
    *             - pBaseFile != nullptr
    */
    AsynchronousAccessFile(
        std::unique_ptr<fs::fsa::IFile>&& pBaseFile,
        fs::OpenMode mode,
        ThreadPool* pThreadPool) NN_NOEXCEPT;

    /**
    * @brief      デストラクタです。
    */
    virtual ~AsynchronousAccessFile() NN_NOEXCEPT NN_OVERRIDE;

private:
    /**
    * @brief      ファイルの内容をバッファに読み込みます。
    *
    * @param[out] outValue    読み込んだデータサイズの格納先
    * @param[in]  offset      読み込み開始位置
    * @param[out] buffer      読み込んだ内容をコピーするバッファ
    * @param[in]  size        読み込むデータサイズ
    * @param[in]  option      オプション
    *
    * @return     関数の処理結果を返します。
    *
    * @pre
    *             - buffer != nullptr
    */
    virtual Result DoRead(
        size_t* outValue,
        int64_t offset,
        void* buffer,
        size_t size,
        const fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;

    Result DoReadImpl(
        size_t* outValue,
        int64_t offset,
        void* buffer,
        size_t size,
        const fs::ReadOption& option) NN_NOEXCEPT;

    /**
    * @brief      バッファの内容をストレージに書き込みます。
    *
    * @param[in]  offset  書き込み開始位置
    * @param[in]  buffer  書き込むデータ
    * @param[in]  size    書き込むデータサイズ
    * @param[in]  option  オプション
    *
    * @return     関数の処理結果を返します。
    *
    * @pre
    *             - buffer != nullptr
    */
    virtual Result DoWrite(
        int64_t offset,
        const void* buffer,
        size_t size,
        const fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルサイズを変更します。
    *
    * @param[in]    size    変更後のファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルサイズを取得します。
    *
    * @param[out]   outValue    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        特定の範囲に対して操作を実行します。
    *
    * @param[out]   outBuffer       操作の結果を格納するバッファ
    * @param[in]    outBufferSize   操作の結果を格納するバッファのサイズ
    * @param[in]    operationId     実行する操作
    * @param[in]    offset          操作開始位置
    * @param[in]    size            操作するサイズ
    * @param[in]    inBuffer        操作に渡すバッファ
    * @param[in]    inBufferSize    操作に渡すバッファのサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DoOperateRange(
        void* outBuffer,
        size_t outBufferSize,
        fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

private:
    std::unique_ptr<fs::fsa::IFile> m_pBaseFile;
    fs::OpenMode m_Mode;
    ThreadPool* m_pThreadPool;
};

//! 非同期アクセスレイヤ
class AsynchronousAccessStorage : public fs::IStorage
{
    NN_DISALLOW_COPY(AsynchronousAccessStorage);

public:
    /**
    * @brief      コンストラクタです。
    *
    * @param[in]  pBaseStorage    ストレージ
    * @param[in]  pThreadPool     スレッドプール
    *
    * @pre
    *             - pBaseStorage != nullptr
    */
    AsynchronousAccessStorage(
        const std::shared_ptr<fs::IStorage>& pBaseStroage,
        ThreadPool* pThreadPool) NN_NOEXCEPT;

    /**
    * @brief      デストラクタです。
    */
    virtual ~AsynchronousAccessStorage() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief      ストレージの内容をバッファに読み込みます。
    *
    * @param[in]  offset  読み込み開始位置
    * @param[out] buffer  読み込んだ内容をコピーするバッファ
    * @param[in]  size    読み込むデータサイズ
    *
    * @return     関数の処理結果を返します。
    *
    * @pre
    *             - buffer != nullptr
    */
    virtual Result Read(
        int64_t offset,
        void* buffer,
        size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief      バッファの内容をストレージに書き込みます。
    *
    * @param[in]  offset  書き込み開始位置
    * @param[in]  buffer  書き込むデータ
    * @param[in]  size    書き込むデータサイズ
    *
    * @return     関数の処理結果を返します。
    *
    * @pre
    *             - buffer != nullptr
    */
    virtual Result Write(
        int64_t offset,
        const void* buffer,
        size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ストレージサイズを取得します。
    *
    * @param[out]   outValue    ストレージサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ストレージサイズを変更します。
    *
    * @param[in]    size    変更後のストレージサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        特定の範囲に対して操作を実行します。
    *
    * @param[out]   outBuffer       操作の結果を格納するバッファ
    * @param[in]    outBufferSize   操作の結果を格納するバッファのサイズ
    * @param[in]    operationId     実行する操作
    * @param[in]    offset          操作開始位置
    * @param[in]    size            操作するサイズ
    * @param[in]    inBuffer        操作に渡すバッファ
    * @param[in]    inBufferSize    操作に渡すバッファのサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ベースストレージを取得します。
    *
    * @return       ベースストレージを返します。
    */
    const IStorage* GetBaseStorage() const NN_NOEXCEPT
    {
        return m_pBaseStorage.get();
    }

    /**
    * @brief        ベースストレージを再設定します。
    *
    * @return       ベースストレージを再設定します。
    */
    void SetBaseStorage(std::shared_ptr<fs::IStorage> pBaseStorage) NN_NOEXCEPT
    {
        m_pBaseStorage = pBaseStorage;
    }

private:
    Result ReadImpl(
        int64_t offset,
        void* buffer,
        size_t size) NN_NOEXCEPT;

private:
    std::shared_ptr<fs::IStorage> m_pBaseStorage;
    ThreadPool* m_pThreadPool;
};

//! バッファプールからバッファを確保するクラス
class PooledBuffer
{
    NN_DISALLOW_COPY(PooledBuffer);

public:
    /*!
    * @brief       バッファプールから割り当てられるバッファの最大サイズを取得します。
    *
    * @return      バッファプールから割り当てられるバッファの最大サイズを返します。
    */
    static size_t GetAllocatableSizeMax() NN_NOEXCEPT
    {
        return GetAllocatableSizeMaxCore(false);
    }

    /*!
    * @brief       バッファプールから割り当てられるバッファの最大サイズを取得します。
    *
    * @return      バッファプールから割り当てられるバッファの最大サイズを返します。
    *
    * @details     本関数は通常より大きいバッファを確保する場合の最大サイズを取得するために使用します。
    *              バッファプールを枯渇させやすくなるため、
    *              特に必要な場合以外はこのサイズで確保しないようにしてください。
    */
    static size_t GetAllocatableParticularlyLargeSizeMax() NN_NOEXCEPT
    {
        return GetAllocatableSizeMaxCore(true);
    }

    /*!
    * @brief       バッファを割り当てずに初期化します。
    */
    PooledBuffer() NN_NOEXCEPT;

    /*!
    * @brief       バッファの所有権を移動します。
    *
    * @param[in]   other   バッファの所有権を移動するオブジェクト
    *
    * @details     バッファの所有権は新たに構築されたオブジェクトに移動され、
    *              other はバッファを所有していない状態になります。
    */
    explicit PooledBuffer(PooledBuffer&& other) NN_NOEXCEPT;

    /*!
    * @brief       バッファプールからバッファを割り当てます。
    *
    * @param[in]   idealSize    割り当てるバッファの理想サイズ
    * @param[in]   requiredSize 割り当てるバッファの最小サイズ
    *
    * @pre
    *              - バッファプール初期化済み
    *              - requiredSize <= GetAllocatableSizeMax()
    *
    * @details     割り当てに成功するまでブロックします。
    *
    *              割り当てられるサイズは idealSize よりも大きい場合があります。
    *              バッファが不足している場合、割り当てられるサイズは指定したサイズより小さくなることがあります。
    *              ただし requiredSize 以上であることが保証されます。
    */
    PooledBuffer(size_t idealSize, size_t requiredSize) NN_NOEXCEPT;

    /*!
    * @brief       割り当てたバッファを解放します。
    */
    ~PooledBuffer() NN_NOEXCEPT;

    /*!
    * @brief       バッファプールからバッファを割り当てます。
    *
    * @param[in]   idealSize    割り当てるバッファの理想サイズ
    * @param[in]   requiredSize 割り当てるバッファの最小サイズ
    *
    * @pre
    *              - バッファプール初期化済み
    *              - バッファ未割り当て
    *              - requiredSize <= GetAllocatableSizeMax()
    *
    * @details     割り当てに成功するまでブロックします。
    *
    *              割り当てられるサイズは idealSize よりも大きい場合があります。
    *              バッファが不足している場合、割り当てられるサイズは指定したサイズより小さくなることがあります。
    *              ただし requiredSize 以上であることが保証されます。
    */
    void Allocate(size_t idealSize, size_t requiredSize) NN_NOEXCEPT
    {
        return AllocateCore(idealSize, requiredSize, false);
    }

    /*!
    * @brief       バッファプールからバッファを割り当てます。
    *
    * @param[in]   idealSize    割り当てるバッファの理想サイズ
    * @param[in]   requiredize 割り当てるバッファの最小サイズ
    *
    * @pre
    *              - バッファプール初期化済み
    *              - バッファ未割り当て
    *              - requiredSize <= GetAllocatableParticularlyLargeSizeMax()
    *
    * @details     割り当てに成功するまでブロックします。
    *
    *              割り当てられるサイズは idealSize よりも大きい場合があります。
    *              バッファが不足している場合、割り当てられるサイズは指定したサイズより小さくなることがあります。
    *              ただし requiredSize 以上であることが保証されます。
    *
    *              本関数は通常より大きいバッファを確保するために使用します。
    *              バッファプールを枯渇させやすくなるため、
    *              特に必要な場合以外は使用しないでください。
    */
    void AllocateParticularlyLarge(size_t idealSize, size_t requiredSize) NN_NOEXCEPT
    {
        return AllocateCore(idealSize, requiredSize, true);
    }

    /*!
    * @brief       割り当てたバッファを解放します。
    */
    void Deallocate() NN_NOEXCEPT;

    /*!
    * @brief       指定のサイズを確保するのに十分な最小サイズまでバッファを縮小します。
    *
    * @param[in]   idealSize    バッファの理想サイズ
    *
    * @pre
    *              - バッファ割り当て済み
    */
    void Shrink(size_t idealSize) NN_NOEXCEPT;

    /*!
    * @brief       割り当てたバッファを取得します。
    *
    * @pre
    *              - バッファ割り当て済み
    *
    * @details     割り当てたバッファを返します。
    */
    char* GetBuffer() const NN_NOEXCEPT;

    /*!
    * @brief       割り当てたバッファのサイズを取得します。
    *
    * @pre
    *              - バッファ割り当て済み
    *
    * @details     割り当てたバッファのサイズを返します。
    */
    size_t GetSize() const NN_NOEXCEPT;

private:
    static size_t GetAllocatableSizeMaxCore(bool isEnableLargeCapacity) NN_NOEXCEPT;
    void AllocateCore(
        size_t idealSize,
        size_t requiredSize,
        bool isEnableLargeCapacity) NN_NOEXCEPT;

private:
    char* m_Buffer;
    size_t m_Size;
};

/**
* @brief   InitializeBufferPool に渡すバッファのアライメント制約を表す定数です。
*
* @details バッファプールに渡すバッファのアライメントの定数です。
*
*          最小のバッファ割当サイズとしても、割り当てられたバッファの
*          アライメント定数としても使われます。
*/
const size_t BufferPoolAlignment = 4096;

/**
* @brief   InitializeBufferPool に渡す workBuffer のサイズを表す定数です。
*
*/
const size_t BufferPoolWorkSize = 320;


/*!
* @brief       バッファプールを初期化します。
*
* @param[in]   buffer  登録するバッファ
* @param[in]   size    登録するバッファのサイズ
*
* @pre
*              - 未初期化
*              - buffer != nullptr
*              - buffer が BufferPoolAlignment 境界に配置されている
*/
Result InitializeBufferPool(char* buffer, size_t size) NN_NOEXCEPT;

/*!
* @brief       バッファプールを初期化します。
*
* @param[in]   buffer      登録するバッファ
* @param[in]   size        登録するバッファのサイズ
* @param[in]   workBuffer  ワークバッファ
* @param[in]   workSize    ワークバッファのサイズ
*
* @pre
*              - 未初期化
*              - buffer != nullptr
*              - buffer が BufferPoolAlignment 境界に配置されている
*
*
 @details      ワークバッファを外部から渡します。
*/
Result InitializeBufferPool(
    char* buffer,
    size_t size,
    char* workBuffer,
    size_t workSize) NN_NOEXCEPT;

/*!
* @brief       指定のバッファがバッファプールに含まれるかどうか判定します。
*
* @param[in]   buffer  判定するバッファ
*
* @return      指定のバッファがバッファプールに含まれるかどうかを返します。
*
* @pre
*              - buffer != nullptr
*/
bool IsPooledBuffer(const void* buffer) NN_NOEXCEPT;

/*!
* @brief       スレッドプールを登録します。
*
* @param[in]   pThreadPool  登録するスレッドプールのポインタ
*
* @pre
*              - pThreadPool が nullptr または未初期化
*/
void RegisterThreadPool(ThreadPool* pThreadPool) NN_NOEXCEPT;

/*!
* @brief       登録済みのスレッドプールを取得します。
*
* @return      登録済みのスレッドプールのポインタ
*
* @pre
*              - スレッドプール登録済み
*/
ThreadPool* GetRegisteredThreadPool() NN_NOEXCEPT;

/*!
* @brief       バッファプールの空き容量のピークを取得
*
* @return      バッファプールの空き容量のピーク
*/
size_t GetPooledBufferFreeSizePeak() NN_NOEXCEPT;

/*!
* @brief       バッファプールのリトライ回数を取得
*
* @return      バッファプールのリトライ回数
*/
size_t GetPooledBufferRetriedCount() NN_NOEXCEPT;

/*!
* @brief       バッファプールの出し渋り回数を取得
*
* @return      バッファプールの出し渋り回数
*/
size_t GetPooledBufferReduceAllocationCount() NN_NOEXCEPT;

/*!
* @brief       バッファプールの空き容量のピーク/リトライ回数をクリア
*/
void ClearPooledBufferPeak() NN_NOEXCEPT;

}}
