﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/os/os_Mutex.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>

#include <nn/crypto/crypto_AesEncryptor.h>

namespace nn { namespace fssystem {


// AES-XTS 暗号化ストレージレイヤ
class AesXtsStorage : public fs::IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(AesXtsStorage);

public:
    static const size_t AesBlockSize = crypto::AesEncryptor128::BlockSize;
    static const size_t KeySize = AesBlockSize;
    static const size_t IvSize  = AesBlockSize;

private:
    IStorage* const m_pBaseStorage;

    // TODO: 鍵の持ち方を変える
    char m_Key[2][KeySize];

    // ストレージ先頭におけるカウンタ
    char m_Iv[IvSize];

    // ブロックサイズ
    const size_t m_BlockSize;

    // バッファを排他利用する
    os::Mutex m_Mutex;

public:
    /**
        @brief pBaseStorage を pKey, pIv で AES-XTS で暗号化・復号するストレージレイヤ

        @pre
            - keySize == KeySize
            - ivSize == IvSize
            - pBaseStorage は有効な IStorage
            - is_aligned(blockSize, AesBlockSize
            - is_aligned(workBufferSize, blockSize)
    */
    AesXtsStorage(
        IStorage* pBaseStorage,
        const void* pKey1,
        const void* pKey2,
        size_t keySize,
        const void* pIv,
        size_t ivSize,
        size_t blockSize
    ) NN_NOEXCEPT;

    /**
        @pre
            - is_aligned(offset, BlockSize)
            - is_aligned(size,   BlockSize)
    */
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
        @pre
            - is_aligned(offset, BlockSize)
            - is_aligned(size,   BlockSize)
    */
    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
        @pre
            - is_aligned(size,   BlockSize)
    */
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;
};

}}
