﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/dbm/fs_FileObjectTemplate.h>

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief        コンストラクタです。
*
* @details      コンストラクタです。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
FileSystemObjectTemplate<
    TDirectoryName,
    TFileName
>::FileSystemObjectTemplate() NN_NOEXCEPT
    : m_pAllocationTable(nullptr),
      m_BlockSize(0),
      m_BlockSizeShift(0)
{
}

/**
* @brief        ファイルシステムをマウントします。
*
* @param[in]    pAllocTable         アロケーションテーブル
* @param[in]    blockSize           ブロックサイズ
* @param[in]    pDirectoryEntry     ディレクトリエントリー用ストレージ
* @param[in]    pFileEntry          ファイルエントリー用ストレージ
*
* @pre          pAllocTable が NULL ではない。
* @pre          pDirectoryEntry が NULL ではない。
* @pre          pFileEntry が NULL ではない。
*
* @details      ファイルシステムをマウントします。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
void FileSystemObjectTemplate<
         TDirectoryName,
         TFileName
     >::Initialize(
         AllocationTable* pAllocTable,
         uint32_t blockSize,
         BufferedAllocationTableStorage* pDirectoryEntry,
         BufferedAllocationTableStorage* pFileEntry
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocTable);

    m_pAllocationTable = pAllocTable;
    m_BlockSize = blockSize;

    // ブロック数を計算するためのシフト数を計算します。
    m_BlockSizeShift = 0;
    uint32_t sizeShift = blockSize >> 1;
    do
    {
        sizeShift >>= 1;
        m_BlockSizeShift++;
    }
    while( sizeShift > 0 );
    NN_SDK_ASSERT_LESS(0U, m_BlockSizeShift);

    return BaseClass::Initialize(pDirectoryEntry, pFileEntry);
}

/**
* @brief        ファイルシステムをアンマウントします。
*
* @details      ファイルシステムをアンマウントします。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
void FileSystemObjectTemplate<
         TDirectoryName,
         TFileName
     >::Finalize() NN_NOEXCEPT
{
    m_pAllocationTable = nullptr;
    m_BlockSize = 0;
    m_BlockSizeShift = 0;
}

/**
* @brief        ディレクトリを作成します。
*
* @param[in]    pFullPath                   フルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に作成できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がファイルでした。
* @retval       ResultAlreadyExists             同名のファイルまたはディレクトリが存在します。
* @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
* @pre          pFullPath が NULL ではない。
*
* @details      ディレクトリを作成します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileSystemObjectTemplate<
           TDirectoryName,
           TFileName
       >::CreateDirectory(
           const PathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    detail::DirectoryInfo dirInfo;
    std::memset(&dirInfo, 0, sizeof(dirInfo));
    return BaseClass::CreateDirectory(pFullPath, dirInfo);
}

/**
* @brief        ディレクトリを開きます。
*
* @param[out]   outIndex                        ディレクトリ ID
* @param[out]   outDirectory                    ディレクトリオブジェクト
* @param[in]    pFullPath                       フルパス
*
* @retval       ResultSuccess                   正常に開きました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がファイルでした。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultDirectoryNotFound         内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          outIndex が NULL ではない。
* @pre          outDirectory が NULL ではない。
* @pre          pFullPath が NULL ではない。
*
* @details      ディレクトリを開きます。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileSystemObjectTemplate<
           TDirectoryName,
           TFileName
       >::OpenDirectory(
           StorageIndex* outIndex,
           DirectoryObjectTempl* outDirectory,
           const PathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);
    NN_SDK_REQUIRES_NOT_NULL(outDirectory);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 対象ファイル用キ－を生成します。
    typename BaseClass::DirectoryKey parentKey;
    typename BaseClass::DirectoryEntry parentEntry;
    typename BaseClass::DirectoryKey dirKey;
    NN_RESULT_DO(
        BaseClass::FindDirectoryRecursive(
            &parentKey,
            &parentEntry,
            &dirKey,
            pFullPath
        )
    );

    // ディレクトリ情報を取得します。
    detail::DirectoryInfo& dirInfo = outDirectory->GetDirectoryInfo();
    NN_RESULT_DO(BaseClass::GetDirectoryInformationFromKey(outIndex, &dirInfo, dirKey));

    // ディレクトリオブジェクトを初期化します。
    outDirectory->Initialize(this, dirKey);
    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルを作成します。
*
* @param[out]   outIndex                        ファイル ID
* @param[in]    pFullPath                       フルパス
* @param[in]    fo                              ファイル操作オプション
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に作成できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultAlreadyExists             同名のファイルまたはディレクトリが存在します。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
* @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
*
* @pre          outIndex が NULL ではない。
* @pre          pFullPath が NULL ではない。
*
* @details      ファイルを作成します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileSystemObjectTemplate<
           TDirectoryName,
           TFileName
       >::CreateFile(
           StorageIndex* outIndex,
           const PathChar* pFullPath,
           const FileOptionalInfo& fo
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    detail::FileInfo fileInfo;
    fileInfo.fs.index = AllocationTable::IndexEnd;
    fileInfo.fs.size.Set(0);
    fileInfo.fo = fo;
    return BaseClass::CreateFile(outIndex, pFullPath, fileInfo);
}

/**
* @brief        指定したファイルを削除します。
*
* @param[in]    pFullpath   ファイルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に削除できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultFileNotFound              内部データに問題があります。
* @retval       ResultDatabaseKeyNotFound       内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
*
* @pre          マウントしている。
* @pre          pFullPath が NULL ではない。
*
* @details      指定したファイルを削除します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileSystemObjectTemplate<
           TDirectoryName,
           TFileName
       >::DeleteFile(
           const PathChar* pFullPath
      ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAllocationTable);

    // ファイル情報を取得します。
    detail::FileInfo fileInfo;
    NN_RESULT_DO(BaseClass::OpenFile(&fileInfo, pFullPath));

    // アロケーションテーブルからファイルのデータ領域を削除します。
    if( fileInfo.fs.index != AllocationTable::IndexEnd )
    {
        NN_RESULT_DO(m_pAllocationTable->Free(fileInfo.fs.index));
        fileInfo.fs.index = AllocationTable::IndexEnd;
    }

    // ファイルオブジェクトを削除します。
    return BaseClass::DeleteFile(pFullPath);
}

/**
* @brief        ファイルを開きます。
*
* @param[out]   outIndex    ファイル ID
* @param[out]   outFile     ファイルオブジェクト
* @param[in]    pFullPath   フルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に開けました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultFileNotFound              内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          outIndex が NULL ではない。
* @pre          pFullPath が NULL ではない。
*
* @details      ファイルを開きます。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileSystemObjectTemplate<
           TDirectoryName,
           TFileName
       >::OpenFile(
           StorageIndex* outIndex,
           FileObjectTempl* pFile,
           const PathChar* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFile);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 対象ファイル用キ－を生成します。
    typename BaseClass::DirectoryKey parentKey;
    typename BaseClass::DirectoryEntry parentEntry;
    typename BaseClass::FileKey fileKey;
    NN_RESULT_DO(
        BaseClass::FindFileRecursive(
            &parentKey,
            &parentEntry,
            &fileKey,
            pFullPath
        )
    );

    // ファイル情報を取得します。
    detail::FileInfo& fileInfo = pFile->GetFileInfo();
    NN_RESULT_DO(BaseClass::OpenFileFromKey(outIndex, &fileInfo, fileKey));

    // ファイルオブジェクトを初期化します。
    pFile->Initialize(this, fileKey);

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したファイルのアロケーションテーブルのインデックスを取得します。
*
* @param[out]   outIndex                        アロケーションテーブルのインデックス
* @param[in]    pFullPath                       フルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultFileNotFound              内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          outIndex が NULL ではない。
* @pre          pFullPath が NULL ではない。
*
* @details      指定したファイルのアロケーションテーブルのインデックスを取得します。
*               ファイルサイズを利用しない特殊なファイルシステム用の機能です。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileSystemObjectTemplate<
           TDirectoryName,
           TFileName
       >::GetAllocationTableIndex(
           uint32_t* outIndex,
           const char* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    detail::FileInfo fileInfo;
    NN_RESULT_DO(BaseClass::OpenFile(&fileInfo, pFullPath));

    *outIndex = fileInfo.fs.index;

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したファイルのアロケーションテーブルのインデックスを変更します。
*
* @param[in]    index                           アロケーションテーブルのインデックス
* @param[in]    pFullPath                       フルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に変更できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultFileNotFound              内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
*
* @pre          pFullPath が NULL ではない。
*
* @details      ファイルサイズを利用しない特殊なファイルシステム用の機能です。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileSystemObjectTemplate<
           TDirectoryName,
           TFileName
       >::SetAllocationTableIndex(
           uint32_t index,
           const char* pFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 対象ファイル用キ－を生成します。
    typename BaseClass::DirectoryKey parentKey;
    typename BaseClass::DirectoryEntry parentEntry;
    typename BaseClass::FileKey fileKey;
    NN_RESULT_DO(
        BaseClass::FindFileRecursive(
            &parentKey,
            &parentEntry,
            &fileKey,
            pFullPath
        )
    );
    StorageIndex indexStorage;
    detail::FileInfo fileInfo;
    NN_RESULT_DO(BaseClass::OpenFileFromKey(&indexStorage, &fileInfo, fileKey));

    fileInfo.fs.index = index;

    return BaseClass::UpdateFileInformationFromKey(fileKey, fileInfo);
}

/**
* @brief        指定したファイルの識別 ID を取得します。
*
* @param[out]   outId                           ファイル ID
* @param[in]    pFullPath                       フルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultFileNotFound              内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          outId が NULL ではない。
* @pre          pFullPath が NULL ではない。
*
* @details      ファイルサイズを利用しない特殊なファイルシステム用の機能です。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileSystemObjectTemplate<
           TDirectoryName,
           TFileName
       >::GetSessionId(int64_t* outId, const char* pFullPath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outId);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    detail::FileInfo fileInfo;
    NN_RESULT_DO(BaseClass::OpenFile(&fileInfo, pFullPath));
    *outId = fileInfo.fs.size.Get();
    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したファイルに識別 ID を設定します。
*
* @param[in]    id                              ファイル ID
* @param[in]    pFullPath                       フルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に設定できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultFileNotFound              内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
*
* @pre          pFullPath が NULL ではない。
*
* @details      ファイルサイズを利用しない特殊なファイルシステム用の機能です。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileSystemObjectTemplate<
           TDirectoryName,
           TFileName
       >::SetSessionId(int64_t id, const char* pFullPath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 対象ファイル用キ－を生成します。
    typename BaseClass::DirectoryKey parentKey;
    typename BaseClass::DirectoryEntry parentEntry;
    typename BaseClass::FileKey fileKey;
    NN_RESULT_DO(
        BaseClass::FindFileRecursive(
            &parentKey,
            &parentEntry,
            &fileKey,
            pFullPath
        )
    );
    StorageIndex indexStorage;
    detail::FileInfo fileInfo;
    NN_RESULT_DO(BaseClass::OpenFileFromKey(&indexStorage, &fileInfo, fileKey));

    fileInfo.fs.size.Set(id);

    return BaseClass::UpdateFileInformationFromKey(fileKey, fileInfo);
}

/**
* @brief        指定されたサイズを抱合できるブロック数を取得します。
*
* @param[in]    size        サイズ
*
* @return       指定されたサイズを抱合できるブロック数。
*
* @pre          size が 0 以上。
*
* @details      指定されたサイズを抱合できるブロック数を取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
uint32_t FileSystemObjectTemplate<
             TDirectoryName,
             TFileName
         >::RoundUpBlockSize(int64_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(0, size);
    return static_cast<uint32_t>( ( size + m_BlockSize - 1 ) >> m_BlockSizeShift );
}

/**
* @brief        ファイルサイズを変更します。
*
* @param[in]    pFileInfo                   ファイル情報
* @param[in]    fileKey                     ファイルキー
* @param[in]    newSize                     変更後のファイルサイズ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   ファイルサイズが変更できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          fileKey はファイルではなくディレクトリです。
* @retval       ResultFileNotFound              fileKey に紐づくファイルエントリーがありません。
* @retval       ResultOutOfRange                newSize が範囲外です。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       ResultAllocationTableFull       アロケーションテーブルに空きが無い。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
*
* @pre          マウントしている。
* @pre          pFileInfo が NULL ではない。
* @pre          size が 0 以上
* @pre          newSize が 4G x ブロックサイズ未満
*
* @details      ファイルサイズを変更します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileSystemObjectTemplate<
           TDirectoryName,
           TFileName
       >::ResizeFile(
           detail::FileInfo* pFileInfo,
           const typename BaseClass::FileKey& fileKey,
           int64_t newSize
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFileInfo);
    NN_SDK_REQUIRES_NOT_NULL(m_pAllocationTable);
    NN_SDK_REQUIRES_LESS_EQUAL(0, newSize);

    // 4G * ブロックサイズ以上のサイズには対応していません。
    static const int64_t MaxBlockCount = 4LL * 1024 * 1024 * 1024;
    if (MaxBlockCount << m_BlockSizeShift <= newSize)
    {
        return nn::fs::ResultOutOfResource();
    }

    uint32_t newBlockCount = RoundUpBlockSize(newSize);
    uint32_t orgBlockCount = RoundUpBlockSize(pFileInfo->fs.size.Get());
    if( newBlockCount == 0 )
    {
        // ブロックを解放します。
        if( orgBlockCount > 0 )
        {
            NN_RESULT_DO(m_pAllocationTable->Free(pFileInfo->fs.index));
        }
        pFileInfo->fs.index = AllocationTable::IndexEnd;
    }
    else if( orgBlockCount == 0 )
    {
        // 新規ブロックを確保します。
        NN_RESULT_DO(m_pAllocationTable->Allocate(&pFileInfo->fs.index, newBlockCount));
    }
    else if( orgBlockCount > newBlockCount )
    {
        // ファイルサイズが収まるようにブロックを分割します。
        uint32_t indexAfter = 0;
        NN_RESULT_DO(m_pAllocationTable->Split(&indexAfter, pFileInfo->fs.index, newBlockCount));

        // 分割して余った方は解放します。
        NN_RESULT_DO(m_pAllocationTable->Free(indexAfter));
    }
    else if( orgBlockCount < newBlockCount )
    {
        // 足りないブロック数分確保します。
        uint32_t newIndex;
        NN_RESULT_DO(m_pAllocationTable->Allocate(&newIndex, newBlockCount - orgBlockCount));

        // 新しく確保したブロックを結合します。
        NN_RESULT_DO(m_pAllocationTable->Concat(pFileInfo->fs.index, newIndex));
    }
    else
    {
        // ブロック数は変化しませんでした。
    }

    // ファイル情報を更新します。
    pFileInfo->fs.size.Set(newSize);
    NN_RESULT_DO(BaseClass::UpdateFileInformationFromKey(fileKey, *pFileInfo));

    NN_RESULT_SUCCESS;
}

/**
* @brief 指定したデータ領域の次のデータ領域を取得します。
*
* @param[out]   outNextIndex        次の連続ブロックチェインの先頭インデックス。
*                                   最後の連続ブロックチェインであれば INDEX_EMPTY。
* @param[out]   outBlockCount       現在の連続ブロックチェインのデータ領域数
* @param[in]    index               現在の連続ブロックチェインの先頭インデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       正常に取得できました。
* @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
*
* @pre          マウントしている。
* @pre          outNextIndex が NULL ではない。
* @pre          outBlockCount が NULL ではない。
*
* @details      指定したデータ領域の次のデータ領域を取得します。
*               連続ブロックチェイン単位でイテレーションします。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileSystemObjectTemplate<
           TDirectoryName,
           TFileName
       >::GetNextAllocationBlock(
           uint32_t* outNextIndex,
           uint32_t* outBlockCount,
           uint32_t index
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outNextIndex);
    NN_SDK_REQUIRES_NOT_NULL(outBlockCount);
    NN_SDK_REQUIRES_NOT_NULL(m_pAllocationTable);
    return m_pAllocationTable->ReadNext(outNextIndex, outBlockCount, index);
}

/**
* @brief        指定したデータ領域の手前のデータ領域を取得します。
*
* @param[out]   outPreviousIndex    手前の連続ブロックチェインの先頭インデックス。
*                                   最初の連続ブロックチェインであれば INDEX_EMPTY。
* @param[out]   outBlockCount       現在の連続ブロックチェインのデータ領域数
* @param[in]    index               現在の連続ブロックチェインの先頭インデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       正常に取得できました。
* @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
*
* @pre          マウントしている。
* @pre          outPreviousIndex が NULL ではない。
* @pre          outBlockCount が NULL ではない。
*
* @details      指定したデータ領域の手前のデータ領域を取得します。
*               連続ブロックチェイン単位でイテレーションします。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileSystemObjectTemplate<
           TDirectoryName,
           TFileName
       >::GetPrevAllocationBlock(
           uint32_t* outPreviousIndex,
           uint32_t* outBlockCount,
           uint32_t index
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAllocationTable);
    NN_SDK_REQUIRES_NOT_NULL(outPreviousIndex);
    NN_SDK_REQUIRES_NOT_NULL(outBlockCount);
    return m_pAllocationTable->ReadPrevious(outPreviousIndex, outBlockCount, index);
}

}}}
