﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>

#include <nn/util/util_BitUtil.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SubStorage.h>

class BitmapTest;
class BitmapAPITest;
class BitmapExpandTest;
class BitmapAgingTest;

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief ビットマップクラスです。
*/
class Bitmap
{
public:
    //! ビットマップ用イテレータです。
    struct Iterator
    {
        uint32_t index;                 //! 次回探索インデックス
    };
    NN_STATIC_ASSERT(std::is_pod<Iterator>::value);

public:
    /**
    * @brief        指定したブロック数(ビット数)の二重化ビットマップに必要なデータサイズを取得します。
    *
    * @param[in]    bitCount ブロック数
    *
    * @return       指定したブロック数の二重化ビットマップに必要なデータサイズ
    */
    static int64_t QuerySize(uint32_t bitCount) NN_NOEXCEPT
    {
        return nn::util::align_up<uint32_t>(bitCount, 32) / 8;
    }

    /**
    * @brief        ビットマップ用ストレージをフォーマットします。
    *
    * @param[in]    bitCount  総ブロック数(ビット数)
    * @param[in]    storage   ビットマップ用ストレージ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Format(
                      uint32_t bitCount,
                      fs::SubStorage storage
                  ) NN_NOEXCEPT;

    /**
    * @brief        ビットマップ用ストレージを拡張します。
    *
    * @param[in]    bitCountOld   拡張以前のブロック数
    * @param[in]    bitCountNew   拡張後のブロック数
    * @param[in]    storage   ビットマップ用ストレージ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Expand(
                      uint32_t bitCountOld,
                      uint32_t bitCountNew,
                      fs::SubStorage storage
                  ) NN_NOEXCEPT;

    //! コンストラクタ
    Bitmap() NN_NOEXCEPT;

    /**
    * @brief        ビットマップストレージをマウントします。
    *
    * @param[in]    bitCount  総ブロック数(ビット数)
    * @param[in]    storage     ビットマップ用ストレージ
    *
    * @return       関数の処理結果を返します。
    */
    void Initialize(
             uint32_t bitCount,
             fs::SubStorage storage
         ) NN_NOEXCEPT;

    /**
    * @brief        イテレーションを開始します。
    *
    * @param[out]   it      イテレータ
    * @param[in]    index   インデックス
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - it != nullptr
    *               - index <= m_BitCount
    */
    Result IterateBegin(
               Iterator* it,
               uint32_t index
           ) const NN_NOEXCEPT;

    /**
    * @brief        ビットマップに対してイテレーションを行います。
    *
    * @param[out]   outZeroCount    0 が連続している個数
    * @param[out]   outOneCount     1 が連続している個数
    * @param[out]   it              イテレータ
    *
    * @return       関数の処理結果を返します。
    *
    * @details
    *               outZeroCount または outOneCount のどちらかは必ず 0 になります。
    *               イテレーション完了時には両方とも 0 が格納されます。
    *
    * @pre
    *               - outZeroCount != nullptr
    *               - outOneCount != nullptr
    *               - it != nullptr
    */
    Result IterateNext(
               uint32_t* outZeroCount,
               uint32_t* outOneCount,
               Iterator* it
           ) NN_NOEXCEPT;

    /**
    * @brief        ビットマップに対して読み込むサイズを制限したイテレーションを行います。
    *
    * @param[out]       outZeroCount        0 が連続している個数
    * @param[out]       outOneCount         1 が連続している個数
    * @param[in,out]    it                  イテレータ
    * @param[in]        countMaxIterateZero 0 が連続しているかチェックする最大個数（0 で無制限）
    * @param[in]        countMaxIterateOne  1 が連続しているかチェックする最大個数（0 で無制限）
    *
    * @return       関数の処理結果を返します。
    *
    * @details
    *               outZeroCount または outOneCount のどちらかは必ず 0 になります。
    *               イテレーション完了時には両方とも 0 が格納されます。
    *
    * @pre
    *               - outZeroCount != nullptr
    *               - outOneCount != nullptr
    *               - it != nullptr
    */
    Result LimitedIterateNext(
               uint32_t* outZeroCount,
               uint32_t* outOneCount,
               Iterator* it,
               uint32_t countMaxIterateZero,
               uint32_t countMaxIterateOne
           ) NN_NOEXCEPT;

    /**
    * @brief        指定した範囲のビットデータを反転します。
    *
    * @param[in]    index   インデックス
    * @param[in]    count   ビット数
    *
    * @return       関数の処理結果を返します。
    */
    Result Reverse(uint32_t index, uint32_t count) NN_NOEXCEPT;

    /**
    * @brief        ビットデータを0クリアします。
    *
    * @return       関数の処理結果を返します。
    */
    Result Clear() NN_NOEXCEPT;

private:
    /**
    * @brief        ストレージからデータを読み込みします。
    *
    * @param[out]   buffer      読み込んだデータを格納するバッファ
    * @param[in]    maxBytes    読み込み先のバッファサイズ。4 の倍数である必要があります。
    * @param[in]    index       読み込み開始インデックス
    *
    * @return       関数の処理結果を返します。
    *
    * @details
    *               読み込みは 32 ビット単位で行われます。
    *               読込先のバッファサイズは 4 の倍数にしておく必要があります。
    *
    * @pre
    *               - buffer != nullptr
    */
    Result ReadBlock(
               void* buffer,
               uint32_t maxBytes,
               uint32_t index
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したインデックスから 32 ビット分のビットマップを取得します。
    *
    * @param[out]   outValue    ビットマップ
    * @param[in]    index       インデックス
    *
    * @return       関数の処理結果を返します。
    *
    * @details      テスト用の機能です。
    *
    * @pre
    *               - outValue != nullptr
    */
    Result GetBitmap32(
               uint32_t* outValue,
               uint32_t index
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したインデックスのビットが 0 か 1 かを取得します。
    *
    * @param[out]   outValue    1 ならば true、0 ならば false
    * @param[in]    index       インデックス
    *
    * @return       関数の処理結果を返します。
    *
    * @details      テスト用の機能です。
    *
    * @pre
    *               - outValue != nullptr
    */
    Result GetBit(bool* outValue, uint32_t index) NN_NOEXCEPT;

private:
    uint32_t m_BitCount;                //! 総ビット数
    fs::SubStorage m_Storage;           //! ビットマップ用ストレージ

private:
    // テスト用クラスに private 関数アクセスを許可します。
    friend class ::BitmapTest;
    friend class ::BitmapAPITest;
    friend class ::BitmapExpandTest;
    friend class ::BitmapAgingTest;
};

}}} // namespace nn::fssystem::dbm
