﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_MemoryReportInfo.h>

namespace nn { namespace fssrv {

class MemoryReportCreator
{
public:
    class MemoryReport
    {
    public:
        virtual size_t GetFreeSizePeak() NN_NOEXCEPT = 0;
        virtual size_t GetTotalAllocatableSizePeak() NN_NOEXCEPT = 0;
        virtual size_t GetRetriedCount() NN_NOEXCEPT = 0;
        virtual void Clear() NN_NOEXCEPT = 0;
    };
    template<typename PeakFreeSizeFunction, typename PeakTotalAllocatableSizeFunction, typename RetriedCountFunction, typename ClearFunction>
    class FunctionalMemoryReport : public MemoryReport
    {
    public:
        FunctionalMemoryReport(PeakFreeSizeFunction getFreeSizePeak, PeakTotalAllocatableSizeFunction getTotalAllocatableSizePeak, RetriedCountFunction getRetriedCount, ClearFunction clear)
            : m_GetFreeSizePeak(getFreeSizePeak)
            , m_GetTotalAllocatableSizePeak(getTotalAllocatableSizePeak)
            , m_GetRetriedCount(getRetriedCount)
            , m_Clear(clear)
        {
        }
    public:
        virtual size_t GetFreeSizePeak() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_GetFreeSizePeak();
        }
        virtual size_t GetTotalAllocatableSizePeak() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_GetTotalAllocatableSizePeak();
        }
        virtual size_t GetRetriedCount() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_GetRetriedCount();
        }
        virtual void Clear() NN_NOEXCEPT NN_OVERRIDE
        {
            m_Clear();
        }
    private:
        PeakFreeSizeFunction m_GetFreeSizePeak;
        PeakTotalAllocatableSizeFunction m_GetTotalAllocatableSizePeak;
        RetriedCountFunction m_GetRetriedCount;
        ClearFunction m_Clear;
    };

    template<typename PeakFreeSizeFunction, typename PeakTotalAllocatableSizeFunction, typename RetriedCountFunction, typename ClearFunction>
    static FunctionalMemoryReport<PeakFreeSizeFunction, PeakTotalAllocatableSizeFunction, RetriedCountFunction, ClearFunction> CreateFunctionalMemoryReport(PeakFreeSizeFunction getFreeSizePeak, PeakTotalAllocatableSizeFunction getTotalAllocatableSizePeak, RetriedCountFunction getRetriedCount, ClearFunction clear)
    {
        return FunctionalMemoryReport<PeakFreeSizeFunction, PeakTotalAllocatableSizeFunction, RetriedCountFunction, ClearFunction>(getFreeSizePeak, getTotalAllocatableSizePeak, getRetriedCount, clear);
    }

public:
    MemoryReportCreator(
        MemoryReport* pBufferManagerMemoryReport,
        MemoryReport* pExpHeapMemoryReport,
        MemoryReport* pBufferPoolMemoryReport
    ) NN_NOEXCEPT;
    ~MemoryReportCreator() NN_NOEXCEPT;

public:
    nn::Result GetMemoryReportInfo(nn::fs::MemoryReportInfo* pOutMemoryReportInfo) NN_NOEXCEPT;

private:
    MemoryReport* m_pBufferManagerMemoryReport;
    MemoryReport* m_pExpHeapMemoryReport;
    MemoryReport* m_pBufferPoolMemoryReport;
    nn::os::Mutex m_Mutex;
};

/*!
* @brief       MemoryReportCreator インスタンスの設定
*/
void SetMemoryReportCreator(MemoryReportCreator* pMemoryReport) NN_NOEXCEPT;

/*!
* @brief       MemoryReportInfo の取得
*/
nn::Result GetMemoryReportInfo(nn::fs::MemoryReportInfo* pOutMemoryReportInfo) NN_NOEXCEPT;

}}
