﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssrv/fssrv_IFileSystemCreator.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fssystem/buffers/fs_IBufferManager.h>
#include <nn/fssystem/fs_ISaveDataExtraDataAccessor.h>
#include <nn/fssystem/fs_IMacGenerator.h>
#include <nn/fssystem/save/fs_HierarchicalDuplexStorage.h>
#include <nn/fssystem/save/fs_HierarchicalIntegrityVerificationStorage.h>

namespace nn { namespace fssrv { namespace fscreator {

    class SaveDataFileSystemCreator : public ISaveDataFileSystemCreator
    {
        NN_DISALLOW_COPY(SaveDataFileSystemCreator);
        NN_DISALLOW_MOVE(SaveDataFileSystemCreator);
    public:
        struct MacGenerationSeed
        {
            static const int Size = 16;
            char value[Size];
        };
        NN_STATIC_ASSERT(std::is_pod<SaveDataFileSystemCreator::MacGenerationSeed>::value);

    private:
        class DeviceUniqueMacGenerator : public fssystem::IMacGenerator
        {
        public:
            explicit DeviceUniqueMacGenerator(void(*pGenerateMac)(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize))
                : m_pGenerateMac(pGenerateMac)
            {
            }

            virtual void Generate(void* macBuffer, size_t macBufferSize, const void* targetBuffer, size_t targetBufferSize) NN_NOEXCEPT NN_OVERRIDE
            {
                m_pGenerateMac(macBuffer, macBufferSize, targetBuffer, targetBufferSize);
            }

        private:
            void(*m_pGenerateMac)(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize);
        };

        class SeedUniqueMacGenerator : public fssystem::IMacGenerator
        {
        public:
            SeedUniqueMacGenerator(void(*pGenerateMac)(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize, const void* seedBuffer, size_t seedBufferSize), const MacGenerationSeed* pSeed)
                : m_pGenerateMac(pGenerateMac)
                , m_pSeed(pSeed)
            {
            }

            virtual void Generate(void* macBuffer, size_t macBufferSize, const void* targetBuffer, size_t targetBufferSize) NN_NOEXCEPT NN_OVERRIDE
            {
                m_pGenerateMac(macBuffer, macBufferSize, targetBuffer, targetBufferSize, m_pSeed, MacGenerationSeed::Size);
            }

        private:
            void(*m_pGenerateMac)(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize, const void* seedBuffer, size_t seedBufferSize);
            const MacGenerationSeed* m_pSeed;
        };

    private:
        fssystem::IMacGenerator* GetMacGenerator(bool isDeviceUniqueMac)
        {
            return isDeviceUniqueMac ? static_cast<fssystem::IMacGenerator*>(&m_DeviceUniqueMacGenerator)
                                     : static_cast<fssystem::IMacGenerator*>(&m_SeedUniqueMacGenerator);
        }


    public:
        SaveDataFileSystemCreator(
            nn::MemoryResource* pAllocator,
            nn::fssystem::IBufferManager* pBufferManager,
            void(*pGenerateDeviceUniqueMac)(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize),
            void(*pGenerateSeedUniqueMac)(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize, const void* seedBuffer, size_t seedBufferSize),
            void(*pGenerateRandom)(void* pHash, size_t sizeHash),
            uint32_t minimumVersion
        ) NN_NOEXCEPT;

        virtual Result CreateRaw(std::shared_ptr<nn::fs::fsa::IFile>* outValue, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor>* outExtraDataAccessor, fssystem::ISaveDataFileSystemCacheManager* pCacheManager, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, bool isAllowedDirectorySaveData, bool isDeviceUniqueMac, nn::fs::SaveDataType saveDataType, fssystem::SaveDataCommitTimeStampGetter timeStampGetter) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result CreateExtraDataAccessor(std::shared_ptr<fssystem::ISaveDataExtraDataAccessor>* outValue, std::shared_ptr<nn::fs::IStorage> storage, bool isDeviceUniqueMac, bool isTemporaryFs) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result CreateInternalStorage(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor>* outExtraDataAccessor, std::shared_ptr<nn::fssystem::SaveDataFileSystem>&& saveDataFileSystem, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, bool isDeviceUniqueMac, bool isTemporaryFs, fssystem::SaveDataCommitTimeStampGetter timeStampGetter) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result RecoverMasterHeader(std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, fssystem::IBufferManager* pBufferManager, bool isDeviceUniqueMac) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result UpdateMac(std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, bool isDeviceUniqueMac) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Format(
            fs::SubStorage baseStorage,
            size_t sizeBlock,
            int countExpandMax,
            const fssystem::save::HierarchicalDuplexStorageControlArea::InputParam& paramDuplex,
            const fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
            uint32_t countDataBlock,
            uint32_t countReservedBlock,
            fssystem::IBufferManager* pBufferManager,
            bool isDeviceUniqueMac,
            const nn::fs::SaveDataHashSalt& hashSalt
        ) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result FormatAsIntegritySaveData(
            fs::SubStorage baseStorage,
            size_t sizeBlock,
            const fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
            uint32_t countDataBlock,
            fssystem::IBufferManager* pBufferManager,
            bool isDeviceUniqueMac
        ) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result ExtractSaveDataParameters(
            fssystem::save::JournalIntegritySaveDataParameters* outParameters,
            fs::IStorage* pSaveDataStorage,
            bool isDeviceUniqueMac
        ) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result ExtendSaveData(
            detail::SaveDataExtender* pExtender,
            fs::SubStorage saveDataStorage,
            fs::SubStorage contextStorage,
            bool isDeviceUniqueMac
        ) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result ReadExtraData(
            nn::fs::SaveDataExtraData* outData,
            fs::SubStorage baseStorage,
            fssystem::IBufferManager* pBufferManager,
            bool isDeviceUniqueMac,
            bool isTemporaryFs
        ) NN_NOEXCEPT NN_OVERRIDE;

        virtual void SetMacGenerationSeed(const void* pSeed, size_t seedSize) NN_NOEXCEPT NN_OVERRIDE;

    private:
        nn::MemoryResource* const m_Allocator;
        nn::fssystem::IBufferManager* const m_pBufferManager;

        DeviceUniqueMacGenerator m_DeviceUniqueMacGenerator;
        SeedUniqueMacGenerator   m_SeedUniqueMacGenerator; // TORIAEZU: 扱う seed は 1 種のみ
        MacGenerationSeed        m_Seed;
        uint32_t                 m_MinimumVersion;
        fssystem::SaveDataCommitRandomGenerater m_GenerateRandom;
    };

}}}
