﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @file
* @brief    fs モジュール用のアクセスログ出力です。
* @details  fs モジュール用のアクセスログを出力する API の宣言です。
*/

#pragma once

#include <nn/fs/fs_AccessLogPrivate.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Priority.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/fs/fs_Result.h>
#include <nn/os/os_TickApi.h>
#include <nn/os/os_TickTypes.h>

namespace nn { namespace fs { namespace detail {

enum AccessLogTarget : uint32_t
{
    AccessLogTarget_None = 0x0,
    AccessLogTarget_Application = 0x1,
    AccessLogTarget_System = 0x2,
};

struct IdentifyAccessLogHandle
{
    void* handle;
public:
    static IdentifyAccessLogHandle MakeValue(void* pHandle) NN_NOEXCEPT
    {
        return IdentifyAccessLogHandle{ pHandle };
    }
};

bool IsEnabledAccessLog(uint32_t target) NN_NOEXCEPT;
bool IsEnabledAccessLog() NN_NOEXCEPT;

void OutputAccessLog(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::FileHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT;
void OutputAccessLog(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::DirectoryHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT;
void OutputAccessLog(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::detail::IdentifyAccessLogHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT;
void OutputAccessLog(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    const void* pHandle,
    const char* fmt,
    ...
) NN_NOEXCEPT;
void OutputAccessLog(
    nn::Result result,
    nn::fs::Priority priority,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    const void* pHandle,
    const char* fmt,
    ...
) NN_NOEXCEPT;
void OutputAccessLog(
    nn::Result result,
    nn::fs::PriorityRaw priorityRow,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    const void* pHandle,
    const char* fmt,
    ...
) NN_NOEXCEPT;

void OutputAccessLogForPrecondition(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::FileHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT;
void OutputAccessLogForPrecondition(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::DirectoryHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT;
void OutputAccessLogForPrecondition(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    const void* pHandle,
    const char* fmt,
    ...
) NN_NOEXCEPT;
void OutputAccessLogForPrecondition(
    bool result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::FileHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT;
void OutputAccessLogForPrecondition(
    bool result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::DirectoryHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT;
void OutputAccessLogForPrecondition(
    bool result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    const void* pHandle,
    const char* fmt,
    ...
) NN_NOEXCEPT;

bool IsEnabledHandleAccessLog(nn::fs::FileHandle handle) NN_NOEXCEPT;
bool IsEnabledHandleAccessLog(nn::fs::DirectoryHandle handle) NN_NOEXCEPT;
bool IsEnabledHandleAccessLog(nn::fs::detail::IdentifyAccessLogHandle handle) NN_NOEXCEPT;
bool IsEnabledHandleAccessLog(const void* pHandle) NN_NOEXCEPT;

bool IsEnabledFileSystemAccessorAccessLog(const char* mountName) NN_NOEXCEPT;
void EnableFileSystemAccessorAccessLog(const char* mountName) NN_NOEXCEPT;

inline size_t ReferenceQuerySizeOutValue(size_t* pQuerySize) NN_NOEXCEPT
{
    return (pQuerySize == nullptr) ? 0 : *pQuerySize;
}

class IdString
{
public:
    template<typename T>
    const char* ToString(T id) NN_NOEXCEPT;
private:
    const char* ToValueString(int id) NN_NOEXCEPT;
private:
    char m_StringBuffer[32];
};

}}}

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_CLASSNAME(name)      ", class_name: " #name
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_NAMESPACE(name)      ", name_space: " #name

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_SIZE                 ", size: %lld"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_QUERY_SIZE           ", size: %zu"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE      ", offset: %lld, size: %zu"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH                 ", path: \"%s\""
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH_AND_SIZE        ", path: \"%s\", size: %lld"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OPENMODE             ", open_mode: 0x%X"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_RENAME NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH ", new_path: \"%s\""
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT                ", name: \"%s\""
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_INDEX                ", index: %d"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_USERID               ", userid: 0x%016llX%016llX"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_DATAID               ", dataid: 0x%llX"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_SYSTEMDATAID         ", systemdataid: 0x%llX"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_APPLICATIONID        ", applicationid: 0x%llX"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_BISPARTITIONID       ", bispartitionid: %s"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_IMAGEDIRECTORYID     ", imagedirectoryid: %s"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_CLOUDBACKUPWORKSTORAGEID ", cloudbackupworkstorageid: %s"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID           ", savedataid: 0x%llX"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID      ", savedataspaceid: %s"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_PROGRAMID            ", programid: 0x%llX"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_CONTENTTYPE          ", content_type: %s"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FILEHANDLE           ", file_handle: 0x%p"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_CONTENTSTORAGEID     ", contentstorageid: %s"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_GAMECARDHANDLE       ", gamecard_handle: 0x%X"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_GAMECARDPARTITION    ", gamecard_partition: %s"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_CACHESTORAGELISTHANDLE ", cachestoragelist_handle: 0x%p"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_INFOBUFFERCOUNT      ", infobuffercount: 0x%X"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_BUFFERSIZE           ", buffer_size: %zu"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FILESIZE             ", file_size: %lld"

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAOWNERID      ", save_data_owner_id: 0x%llX"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE         ", save_data_size: %lld"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAJOURNALSIZE  ", save_data_journal_size: %lld"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAFLAGS        ", save_data_flags: 0x%08X"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATATIMESTAMP    ", save_data_time_stamp: %lld"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATACOMMITID     ", save_data_commit_id: 0x%016llX"

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_NFP_MODELTYPE        ", model_type: %d"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_NFP_MOUNTTARGET      ", nfp_mount_target: %d"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_NFP_CREATEINFO_ACCESSID ", access_id: 0x%08X"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_NFP_CREATEINFO_DATASIZE ", data_size: %d"

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEDATASIZE  ", image_data_size: %zu"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEWIDTH     ", width: %d"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEHEIGHT    ", height: %d"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEORIENTATION ", image_orientation: %d"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_REPORTOPTION   ", album_report_option: 0x%X"

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_DIRECTORY_FILENAME   ", path: \"%s/%s\""

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_START        ", fetched: ["
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_1            "{ offset: %lld, size: %zu }"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_2            NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_1 ", { offset: %lld, size: %zu }"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_3            NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_2 ", { offset: %lld, size: %zu }"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_4            NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_3 ", { offset: %lld, size: %zu }"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_5            NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_4 ", { offset: %lld, size: %zu }"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_6            NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_5 ", { offset: %lld, size: %zu }"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_7            NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_6 ", { offset: %lld, size: %zu }"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_8            NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_7 ", { offset: %lld, size: %zu }"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_END          "]"
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCH_EXCEEDED       ", fetch_exceeded: true"


#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTADDONCONTENT(name, index) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_INDEX, name, index

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTSYSTEMDATA(name, systemDataId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_SYSTEMDATAID, name, systemDataId.value

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTSAVEDATA(name, userId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_USERID, name, (userId)._data[0], (userId)._data[1]
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTSAVEDATA_APPLICATIONID(name, applicationId, userId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_APPLICATIONID NN_DETAIL_FS_ACCESS_LOG_FORMAT_USERID, name, applicationId.value, (userId)._data[0], (userId)._data[1]

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCACHESTORAGE_INDEX(name, index) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_INDEX , name, index

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCACHESTORAGE_APPLICATIONID(name, applicationId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_APPLICATIONID , name, applicationId.value

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCACHESTORAGE_APPLICATIONIDINDEX(name, applicationId, index) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_APPLICATIONID NN_DETAIL_FS_ACCESS_LOG_FORMAT_INDEX, name, applicationId.value, index

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTDEVICESAVEDATA_APPLICATIONID(name, applicationId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_APPLICATIONID , name, applicationId.value

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTAPPLICATIONPACKAGE(name, path) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, name, path

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTBCATSAVEDATA(name, applicationId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_APPLICATIONID, name, applicationId.value

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTBIS(name, bisPartitionId, path) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_BISPARTITIONID NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, name, ::nn::fs::detail::IdString().ToString(bisPartitionId), path

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCODE(name, path, programId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH NN_DETAIL_FS_ACCESS_LOG_FORMAT_PROGRAMID, name, path, programId.value

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCONTENT_PATH(name, path, contentType) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH NN_DETAIL_FS_ACCESS_LOG_FORMAT_CONTENTTYPE, name, path, ::nn::fs::detail::IdString().ToString(contentType)
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCONTENT_PROGRAMID(name, programId, contentType) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_PROGRAMID NN_DETAIL_FS_ACCESS_LOG_FORMAT_CONTENTTYPE, name, programId.value, ::nn::fs::detail::IdString().ToString(contentType)
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCONTENT_PATH_AND_PROGRAMID(name, path, programId, contentType) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH NN_DETAIL_FS_ACCESS_LOG_FORMAT_PROGRAMID NN_DETAIL_FS_ACCESS_LOG_FORMAT_CONTENTTYPE, name, path, programId.value, ::nn::fs::detail::IdString().ToString(contentType)
#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCONTENT_PATH_AND_DATAID(name, path, dataId, contentType) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH NN_DETAIL_FS_ACCESS_LOG_FORMAT_DATAID NN_DETAIL_FS_ACCESS_LOG_FORMAT_CONTENTTYPE, name, path, dataId.value, ::nn::fs::detail::IdString().ToString(contentType)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCONTENTSTORAGE(name, contentStorageId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_CONTENTSTORAGEID, name, ::nn::fs::detail::IdString().ToString(contentStorageId)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTGAMECARDPARTITION(name, handle, partition) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_GAMECARDHANDLE NN_DETAIL_FS_ACCESS_LOG_FORMAT_GAMECARDPARTITION, name, handle, ::nn::fs::detail::IdString().ToString(partition)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTLOGO(name, path, programId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH NN_DETAIL_FS_ACCESS_LOG_FORMAT_PROGRAMID, name, path, programId.value

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTSYSTEMSAVEDATA(name, spaceId, systemSaveDataId, userId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID NN_DETAIL_FS_ACCESS_LOG_FORMAT_USERID \
        , name, ::nn::fs::detail::IdString().ToString(spaceId), systemSaveDataId, (userId)._data[0], (userId)._data[1]

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTSYSTEMBCATSAVEDATA(name, systemBcatSaveDataId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID \
        , name, systemBcatSaveDataId

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTROMONFILE(name, fileHandle) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_FILEHANDLE, name, fileHandle.handle

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTIMAGEDIRECTORY(name, imageDirectoryId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_IMAGEDIRECTORYID, name, ::nn::fs::detail::IdString().ToString(imageDirectoryId)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCLOUDBACKUPWORKSTORAGE(name, cloudBackupWorkStorageId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT NN_DETAIL_FS_ACCESS_LOG_FORMAT_CLOUDBACKUPWORKSTORAGEID, name, ::nn::fs::detail::IdString().ToString(cloudBackupWorkStorageId)


#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_EXTENDSAVEDATA(userId, saveDataSize, saveDataJournalSize) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_USERID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAJOURNALSIZE \
    , (userId)._data[0], (userId)._data[1], saveDataSize, saveDataJournalSize

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_GETSAVEDATASIZE(userId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_USERID \
    , (userId)._data[0], (userId)._data[1]

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATECACHESTORAGE(index, size, journalSize) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_INDEX \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAJOURNALSIZE \
    , index, size, journalSize

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATESAVEDATA(applicationId, userId, saveDataOwnerId, size, journalSize, flags) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_APPLICATIONID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_USERID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAOWNERID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAJOURNALSIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAFLAGS \
    , applicationId, (userId)._data[0], (userId)._data[1], saveDataOwnerId, size, journalSize, flags

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATEBCATSAVEDATA(applicationId, size) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_APPLICATIONID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE \
    , applicationId, size

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATEDEVICESAVEDATA(applicationId, saveDataOwnerId, size, journalSize, flags) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_APPLICATIONID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAOWNERID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAJOURNALSIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAFLAGS \
    , applicationId, saveDataOwnerId, size, journalSize, flags

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATETEMPORARYSTORAGE(applicationId, saveDataOwnerId, size, flags) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_APPLICATIONID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAOWNERID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAFLAGS \
    , applicationId, saveDataOwnerId, size, flags

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_SYSTEM_CREATECACHESTORAGE(applicationId, spaceId, saveDataOwnerId, size, journalSize, flags) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_APPLICATIONID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAOWNERID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAJOURNALSIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAFLAGS \
    , applicationId, ::nn::fs::detail::IdString().ToString(spaceId), saveDataOwnerId, size, journalSize, flags

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATESYSTEMSAVEDATA(saveDataSpaceId, systemSaveDataId, userId, saveDataOwnerId, size, journalSize, flags) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_USERID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAOWNERID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAJOURNALSIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAFLAGS \
    , ::nn::fs::detail::IdString().ToString(saveDataSpaceId), systemSaveDataId, userId._data[0], userId._data[1], saveDataOwnerId, size, journalSize, flags

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATESYSTEMBCATSAVEDATA(systemBcatSaveDataId, size, flags) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAFLAGS \
    , systemBcatSaveDataId, size, flags

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_DELETE_SAVEDATA(spaceId, saveDataId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID \
    , ::nn::fs::detail::IdString().ToString(spaceId), saveDataId

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_DELETE_SYSTEMSAVEDATA(saveDataSpaceId, systemSaveDataId, userId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_USERID \
    , ::nn::fs::detail::IdString().ToString(saveDataSpaceId), systemSaveDataId, (userId)._data[0], (userId)._data[1]

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_READCACHESTORAGELIST(handle, infoBufferCount) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_CACHESTORAGELISTHANDLE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_INFOBUFFERCOUNT \
    , handle.handle, infoBufferCount

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_ENABLEINDIVIDUALFILEDATACACHE(path, bufferSize, fileSize) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_BUFFERSIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FILESIZE \
    , path, bufferSize, fileSize


#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_NFP_MOUNT_WITHTARGET(modelType, mountTarget) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_NFP_MODELTYPE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_NFP_MOUNTTARGET \
    , modelType, mountTarget

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_NFP_CREATEAPPLICATIONAREA(createInfo) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_NFP_CREATEINFO_ACCESSID \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_NFP_CREATEINFO_DATASIZE \
    , (createInfo).accessId, (createInfo).initialDataSize


#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_SCREENSHOT(imageDataSize, imageSize, reportOption) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEDATASIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEWIDTH \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEHEIGHT \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_REPORTOPTION \
    , imageDataSize, nn::album::GetImageWidth(imageSize), nn::album::GetImageHeight(imageSize), reportOption

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_SCREENSHOT_ORIENTATION(imageDataSize, imageSize, imageOrientation, reportOption) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEDATASIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEWIDTH \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEHEIGHT \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEORIENTATION \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_REPORTOPTION \
    , imageDataSize, nn::album::GetImageWidth(imageSize), nn::album::GetImageHeight(imageSize), imageOrientation, reportOption

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_SCREENSHOT_UID(imageDataSize, imageSize, imageOrientation, reportOption, userId) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEDATASIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEWIDTH \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEHEIGHT \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_IMAGEORIENTATION \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_REPORTOPTION \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_USERID \
    , imageDataSize, nn::album::GetImageWidth(imageSize), nn::album::GetImageHeight(imageSize), imageOrientation, reportOption, (userId)._data[0], (userId)._data[1]


#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_WRITEFILE_WITH_FLUSH NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE ", write_option: Flush"


#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_0(theOffset, theSize, cacheAccessResult) \
    , theOffset, theSize

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_1(theOffset, theSize, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_0(theOffset, theSize, cacheAccessResult) \
    , cacheAccessResult.GetCacheFetchedRegion(0).offset, cacheAccessResult.GetCacheFetchedRegion(0).size

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_2(theOffset, theSize, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_1(theOffset, theSize, cacheAccessResult) \
    , cacheAccessResult.GetCacheFetchedRegion(1).offset, cacheAccessResult.GetCacheFetchedRegion(1).size

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_3(theOffset, theSize, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_2(theOffset, theSize, cacheAccessResult) \
    , cacheAccessResult.GetCacheFetchedRegion(2).offset, cacheAccessResult.GetCacheFetchedRegion(2).size

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_4(theOffset, theSize, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_3(theOffset, theSize, cacheAccessResult) \
    , cacheAccessResult.GetCacheFetchedRegion(3).offset, cacheAccessResult.GetCacheFetchedRegion(3).size

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_5(theOffset, theSize, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_4(theOffset, theSize, cacheAccessResult) \
    , cacheAccessResult.GetCacheFetchedRegion(4).offset, cacheAccessResult.GetCacheFetchedRegion(4).size

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_6(theOffset, theSize, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_5(theOffset, theSize, cacheAccessResult) \
    , cacheAccessResult.GetCacheFetchedRegion(5).offset, cacheAccessResult.GetCacheFetchedRegion(5).size

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_7(theOffset, theSize, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_6(theOffset, theSize, cacheAccessResult) \
    , cacheAccessResult.GetCacheFetchedRegion(6).offset, cacheAccessResult.GetCacheFetchedRegion(6).size

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_8(theOffset, theSize, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_7(theOffset, theSize, cacheAccessResult) \
    , cacheAccessResult.GetCacheFetchedRegion(7).offset, cacheAccessResult.GetCacheFetchedRegion(7).size

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_0(offset, size, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_START \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_END \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_0(offset, size, cacheAccessResult)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_1(offset, size, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_START \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_1 \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_END \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_1(offset, size, cacheAccessResult)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_2(offset, size, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_START \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_2 \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_END \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_2(offset, size, cacheAccessResult)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_3(offset, size, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_START \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_3 \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_END \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_3(offset, size, cacheAccessResult)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_4(offset, size, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_START \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_4 \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_END \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_4(offset, size, cacheAccessResult)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_5(offset, size, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_START \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_5 \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_END \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_5(offset, size, cacheAccessResult)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_6(offset, size, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_START \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_6 \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_END \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_6(offset, size, cacheAccessResult)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_7(offset, size, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_START \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_7 \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_END \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_7(offset, size, cacheAccessResult)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_8(offset, size, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_START \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_8 \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_END \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_8(offset, size, cacheAccessResult)

#define NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_GE_9(offset, size, cacheAccessResult) \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_START \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_8 \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCHED_END \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_FETCH_EXCEEDED \
    NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_SIZE_CACHE_ACCESS_RESULT_ARG_8(offset, size, cacheAccessResult)


#define NN_DETAIL_FS_ACCESS_LOG_(expression, handle, isEnabled, functionName, ...) \
    [&](const char* apiName) NN_NOEXCEPT { \
        if( !(isEnabled) ) \
        { \
            return (expression); \
        } \
        else \
        { \
            const ::nn::os::Tick start = ::nn::os::GetSystemTick(); \
            auto nnFsResultForAccessLog = (expression); \
            const ::nn::os::Tick end = ::nn::os::GetSystemTick(); \
            ::nn::fs::detail::OutputAccessLog(nnFsResultForAccessLog, start, end, apiName, handle, __VA_ARGS__); \
            return nnFsResultForAccessLog; \
        }\
    }(functionName)

#define NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME_(nnFsResultForAccessLog, start, end, handle, isEnabled, functionName, ...) \
    [&](const char* apiName) NN_NOEXCEPT { \
        if( !(isEnabled) ) \
        { \
            return (nnFsResultForAccessLog); \
        } \
        else \
        { \
            ::nn::fs::detail::OutputAccessLog(nnFsResultForAccessLog, start, end, apiName, handle, __VA_ARGS__); \
            return nnFsResultForAccessLog; \
        }\
    }(functionName)

#define NN_DETAIL_FS_ACCESS_LOG_FOR_PRECONDITION_(expression, handle, isEnabled, functionName, ...) \
    [&](const char* apiName) NN_NOEXCEPT { \
        if( !(isEnabled) ) \
        { \
            return (expression); \
        } \
        else \
        { \
            const ::nn::os::Tick start = ::nn::os::GetSystemTick(); \
            auto result = (expression); \
            const ::nn::os::Tick end = ::nn::os::GetSystemTick(); \
            ::nn::fs::detail::OutputAccessLogForPrecondition(result, start, end, apiName, handle, __VA_ARGS__); \
            return result; \
        }\
    }(functionName)

#define NN_DETAIL_FS_ACCESS_LOG_WITH_PRIORITY_(expression, handle, priority, isEnabled, functionName, ...) \
    [&](const char* apiName) NN_NOEXCEPT { \
        if( !(isEnabled) ) \
        { \
            return (expression); \
        } \
        else \
        { \
            const ::nn::os::Tick start = ::nn::os::GetSystemTick(); \
            auto nnFsResultForAccessLog = (expression); \
            const ::nn::os::Tick end = ::nn::os::GetSystemTick(); \
            ::nn::fs::detail::OutputAccessLog(nnFsResultForAccessLog, priority, start, end, apiName, handle, __VA_ARGS__); \
            return nnFsResultForAccessLog; \
        } \
    }(functionName)

#define NN_DETAIL_FS_ACCESS_LOG_OPENCACHESTORAGELIST_FOR_PRECONDITION(expression, ...) \
    NN_DETAIL_FS_ACCESS_LOG_FOR_PRECONDITION_((expression), nullptr, \
        ::nn::fs::detail::IsEnabledAccessLog(::nn::fs::detail::AccessLogTarget_System), \
        NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

// handle からファイルシステムを解決して、そのアクセスログが有効か判別します
#define NN_DETAIL_FS_ACCESS_LOG(expression, handle, ...) \
    NN_DETAIL_FS_ACCESS_LOG_((expression), handle, \
        ::nn::fs::detail::IsEnabledAccessLog() && ::nn::fs::detail::IsEnabledHandleAccessLog(handle), \
        NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_FUNCTION_NAME(expression, handle, functionName, ...) \
    NN_DETAIL_FS_ACCESS_LOG_((expression), handle, \
        ::nn::fs::detail::IsEnabledAccessLog() && ::nn::fs::detail::IsEnabledHandleAccessLog(handle), \
        functionName, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_RECORD_TIME(id) \
    const ::nn::os::Tick id = ::nn::os::GetSystemTick()

#define NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME(result, start, end, handle, functionName, ...) \
    NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME_((result), start, end, handle, \
        ::nn::fs::detail::IsEnabledAccessLog() && ::nn::fs::detail::IsEnabledHandleAccessLog(handle), \
        functionName, __VA_ARGS__)

// AccessLogTarget_System なアクセスログの場合は、こちらを使用します
#define NN_DETAIL_FS_ACCESS_LOG_SYSTEM(expression, handle, ...) \
    NN_DETAIL_FS_ACCESS_LOG_((expression), handle, \
        ::nn::fs::detail::IsEnabledAccessLog(::nn::fs::detail::AccessLogTarget_System) && ::nn::fs::detail::IsEnabledHandleAccessLog(handle), \
        NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

// ファイルシステムから、そのアクセスログが有効か判別します
#define NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(expression, handle, pFileSystem, ...) \
    NN_DETAIL_FS_ACCESS_LOG_(expression, handle, \
        ::nn::fs::detail::IsEnabledAccessLog() && pFileSystem->IsEnabledAccessLog(), \
        NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_EXPLICIT_FUNCTION_NAME(expression, handle, pFileSystem, functionName, ...) \
    NN_DETAIL_FS_ACCESS_LOG_(expression, handle, \
        ::nn::fs::detail::IsEnabledAccessLog() && pFileSystem->IsEnabledAccessLog(), \
        functionName, __VA_ARGS__)

// ファイルシステムの取得処理用のアクセスログ出力マクロです
#define NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(expression, ...) \
    NN_DETAIL_FS_ACCESS_LOG_FOR_PRECONDITION_((expression), nullptr, \
        ::nn::fs::detail::IsEnabledAccessLog(), \
        NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM_EXPLICIT_FUNCTION_NAME(expression, functionName, ...) \
    NN_DETAIL_FS_ACCESS_LOG_FOR_PRECONDITION_((expression), nullptr, \
        ::nn::fs::detail::IsEnabledAccessLog(), \
        functionName, __VA_ARGS__)


#define NN_DETAIL_FS_ACCESS_LOG_UNMOUNT(expression, name, ...) \
    NN_DETAIL_FS_ACCESS_LOG_((expression), nullptr, \
        ::nn::fs::detail::IsEnabledAccessLog() && ::nn::fs::detail::IsEnabledFileSystemAccessorAccessLog(name), \
        NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_MOUNT_FOR_PRECONDITION(expression, name, ...) \
    NN_DETAIL_FS_ACCESS_LOG_FOR_PRECONDITION_((expression), nullptr, \
        ::nn::fs::detail::IsEnabledAccessLog(::nn::fs::detail::AccessLogTarget_Application), \
        NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_MOUNT(expression, name, ...) \
    NN_DETAIL_FS_ACCESS_LOG_((expression), nullptr, \
        ::nn::fs::detail::IsEnabledAccessLog(::nn::fs::detail::AccessLogTarget_Application), \
        NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE_(name, isEnabled) \
    do \
    { \
        if( isEnabled ) \
        { \
            ::nn::fs::detail::EnableFileSystemAccessorAccessLog(name); \
        } \
    } while( NN_STATIC_CONDITION(0) )

#define NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name) \
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE_(name, ::nn::fs::detail::IsEnabledAccessLog(::nn::fs::detail::AccessLogTarget_Application))

#if !defined(NN_SDK_BUILD_RELEASE)

#define NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT_FOR_PRECONDITION(expression, name, ...) \
    NN_DETAIL_FS_ACCESS_LOG_FOR_PRECONDITION_((expression), nullptr, \
        ::nn::fs::detail::IsEnabledAccessLog(::nn::fs::detail::AccessLogTarget_System), \
        NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(expression, name, ...) \
    NN_DETAIL_FS_ACCESS_LOG_((expression), nullptr, \
        ::nn::fs::detail::IsEnabledAccessLog(::nn::fs::detail::AccessLogTarget_System), \
        NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name) \
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE_(name, ::nn::fs::detail::IsEnabledAccessLog(::nn::fs::detail::AccessLogTarget_System))

#else

#define NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT_FOR_PRECONDITION(expression, ...) expression
#define NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(expression, ...) expression
#define NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(...)

#endif

#define NN_DETAIL_FS_ACCESS_LOG_WITH_PRIORITY(expression, priority, ...) \
    NN_DETAIL_FS_ACCESS_LOG_WITH_PRIORITY_(expression, nullptr, priority, \
    ::nn::fs::detail::IsEnabledAccessLog(), NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_WITH_PRIORITY_SYSTEM(expression, priorityRaw, ...) \
    NN_DETAIL_FS_ACCESS_LOG_WITH_PRIORITY_(expression, nullptr, priorityRaw, \
    ::nn::fs::detail::IsEnabledAccessLog(::nn::fs::detail::AccessLogTarget_System), NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_FILE_DATA_CACHE(expression, ...) \
    NN_DETAIL_FS_ACCESS_LOG_((expression), nullptr, \
        ::nn::fs::detail::IsEnabledAccessLog(), \
        NN_CURRENT_FUNCTION_NAME, __VA_ARGS__)

#if defined(NN_DETAIL_FS_ACCESS_LOG_DETECT_FOR_TEST)

#undef NN_DETAIL_FS_ACCESS_LOG_
#undef NN_DETAIL_FS_ACCESS_LOG_SYSTEM
#undef NN_DETAIL_FS_ACCESS_LOG_MOUNT
#undef NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT
#undef NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE_
#undef NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE
#undef NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE

#define NN_DETAIL_FS_ACCESS_LOG_DETECT_MESSAGE(type, type_detail, functionName, line, expression, name_or_handle, ...) \
    @FS_ACCESS, type, type_detail, functionName, line, expression, name_or_handle, __VA_ARGS__ @

#define NN_DETAIL_FS_ACCESS_LOG_(expression, handle, isEnabled, functionName, ...)    \
    NN_DETAIL_FS_ACCESS_LOG_DETECT_MESSAGE(access, application, functionName, __LINE__, (expression), handle, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_RESULT_AND_TIME_(nnFsResultForAccessLog, start, end, handle, isEnabled, functionName, ...) \
    NN_DETAIL_FS_ACCESS_LOG_DETECT_MESSAGE(access, application, functionName, __LINE__, (nnFsResultForAccessLog), handle, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_SYSTEM(expression, handle, ...) \
    NN_DETAIL_FS_ACCESS_LOG_DETECT_MESSAGE(access, system, NN_CURRENT_FUNCTION_NAME, __LINE__, (expression), handle, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_MOUNT_(expression, name, type_detail, ...) \
    NN_DETAIL_FS_ACCESS_LOG_DETECT_MESSAGE(mount, type_detail, NN_CURRENT_FUNCTION_NAME, __LINE__, (expression), name, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_MOUNT(expression, name, ...) \
    NN_DETAIL_FS_ACCESS_LOG_MOUNT_((expression), name, application, __VA_ARGS__)

#define NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(expression, name, ...) \
    NN_DETAIL_FS_ACCESS_LOG_MOUNT_((expression), name, system, __VA_ARGS__)


#define NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE_(name, type_detail) \
    NN_DETAIL_FS_ACCESS_LOG_DETECT_MESSAGE(enable, type_detail, NN_CURRENT_FUNCTION_NAME, __LINE__, EXP, name)

#define NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name) \
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE_(name, application)

#define NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name) \
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE_(name, system)

#endif
