﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Device Tree の stringlist に関する API の宣言
 */

#pragma once

#include <algorithm>
#include <cstring>

#include <nn/nn_Common.h>

namespace nn { namespace dt {

//! @name stringlist に関する API
//! @{

/**
 * @brief   Device Tree の stringlist を扱うためのユーティリティクラスです。
 * @detail
 *  Device Tree の stringlist を扱うためのユーティリティクラスです。
 *  std::forward_list ライクなインターフェースを実装していますが、以下の制約があります。
 *  - リストの内容は変更不可（関連するメンバは非実装）
 *  -- これに伴い、const でない iterator や reference および before_begin() 等も非実装
 *  - アロケータは使用不可（関連するメンバは非実装）
 *  - コンストラクタでリストの内容を決定する必要がある（そうでないコンストラクタは非実装）
 *  - その他、一部のメンバが非実装
 */
class StringForwardList
{
public:
    using value_type = const char*;
    using size_type = size_t;
    using difference_type = ptrdiff_t;
    using const_reference = const value_type&;

    /**
    * @brief    constイテレータです。
    */
    class const_iterator
    {
    public:
        using value_type = StringForwardList::value_type;
        using difference_type = StringForwardList::difference_type;
        using pointer = const char*;
        using reference = StringForwardList::const_reference;
        using iterator_category = std::forward_iterator_tag;

        /**
        * @brief コンストラクタです。
        * @param[in] オブジェクトで扱う stringlist の先頭ポインタ
        */
        explicit const_iterator(const char* pBody) NN_NOEXCEPT
        : m_pBody(pBody)
        {
        }

        const_iterator(const const_iterator& iter) NN_NOEXCEPT
            : m_pBody(iter.m_pBody)
        {
        }

        const_iterator& operator++() NN_NOEXCEPT
        {
            m_pBody += (std::strlen(m_pBody) + 1);
            return *this;
        }

        const_iterator operator++(int) NN_NOEXCEPT
        {
            const_iterator temporary(*this);
            ++(*this);
            return temporary;
        }

        reference operator*() const NN_NOEXCEPT
        {
            return m_pBody;
        }

        // 二項演算子オーバーロード
        friend bool operator==(const const_iterator& lhs, const const_iterator& rhs) NN_NOEXCEPT
        {
            return *lhs == *rhs;
        }

        friend bool operator!=(const const_iterator& lhs, const const_iterator& rhs) NN_NOEXCEPT
        {
            return !(lhs == rhs);
        }

        friend bool operator<(const const_iterator& lhs, const const_iterator& rhs) NN_NOEXCEPT
        {
            return *lhs < *rhs;
        }

        friend bool operator>(const const_iterator& lhs, const const_iterator& rhs) NN_NOEXCEPT
        {
            return rhs < lhs;
        }

        friend bool operator<=(const const_iterator& lhs, const const_iterator& rhs) NN_NOEXCEPT
        {
            return !(rhs < lhs);
        }

        friend bool operator>=(const const_iterator& lhs, const const_iterator& rhs) NN_NOEXCEPT
        {
            return !(lhs < rhs);
        }
    private:
        const char* m_pBody;
    };

    // コンストラクタ
    StringForwardList(const char* first, const char* last) NN_NOEXCEPT
        : m_pFirst(first)
        , m_pLast(last)
    {
    }

    StringForwardList(const StringForwardList&) NN_NOEXCEPT = default;
    StringForwardList(StringForwardList&&) NN_NOEXCEPT = default;

    // デストラクタ
    ~StringForwardList() NN_NOEXCEPT = default;

    // 代入演算子
    StringForwardList& operator=(const StringForwardList&) NN_NOEXCEPT = default;
    StringForwardList& operator=(StringForwardList&&) NN_NOEXCEPT = default;

    // コンテナの再代入
    void assign(const char* first, const char* last) NN_NOEXCEPT
    {
        m_pFirst = first;
        m_pLast = last;
    }

    // イテレータ
    const_iterator begin() const NN_NOEXCEPT
    {
        return const_iterator(m_pFirst);
    }

    const_iterator end() const NN_NOEXCEPT
    {
        return const_iterator(m_pLast);
    }

    const_iterator cbegin() const NN_NOEXCEPT
    {
        return const_iterator(m_pFirst);
    }

    const_iterator cend() const NN_NOEXCEPT
    {
        return const_iterator(m_pLast);
    }

    bool empty() NN_NOEXCEPT
    {
        return m_pFirst == m_pLast;
    }

    const_reference front() const NN_NOEXCEPT
    {
        return m_pFirst;
    }

    // 二項演算子オーバーロード
    friend bool operator==(const StringForwardList& lhs, const StringForwardList& rhs) NN_NOEXCEPT
    {
        // StringForwardList 自体が不正なら false を返す
        if (lhs.m_pFirst > lhs.m_pLast || rhs.m_pFirst > rhs.m_pLast)
        {
            return false;
        }

        size_t lhsLength = lhs.m_pLast - lhs.m_pFirst;
        size_t rhsLength = rhs.m_pLast - rhs.m_pFirst;

        if (lhsLength == rhsLength)
        {
            return std::memcmp(lhs.m_pFirst, rhs.m_pFirst, lhsLength) == 0;
        }
        else
        {
            return false;
        }
    }

    friend bool operator!=(const StringForwardList& lhs, const StringForwardList& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

    friend bool operator<(const StringForwardList& lhs, const StringForwardList& rhs) NN_NOEXCEPT
    {
        // StringForwardList 自体が不正なら false を返す
        if (lhs.m_pFirst > lhs.m_pLast || rhs.m_pFirst > rhs.m_pLast)
        {
            return false;
        }

        size_t lhsLength = lhs.m_pLast - lhs.m_pFirst;
        size_t rhsLength = rhs.m_pLast - rhs.m_pFirst;

        size_t compareLength = std::min(lhsLength, rhsLength);

        int n = std::memcmp(lhs.m_pFirst, rhs.m_pFirst, compareLength);

        if (n == 0)
        {
            return (lhsLength < rhsLength);
        }
        else
        {
            return (n < 0);
        }
    }

    friend bool operator>(const StringForwardList& lhs, const StringForwardList& rhs) NN_NOEXCEPT
    {
        return rhs < lhs;
    }

    friend bool operator<=(const StringForwardList& lhs, const StringForwardList& rhs) NN_NOEXCEPT
    {
        return !(rhs < lhs);
    }

    friend bool operator>=(const StringForwardList& lhs, const StringForwardList& rhs) NN_NOEXCEPT
    {
        return !(lhs < rhs);
    }
private:
    const char* m_pFirst;
    const char* m_pLast;
};

NN_FORCEINLINE void swap(StringForwardList& lhs, StringForwardList& rhs) NN_NOEXCEPT
{
    StringForwardList temporary(lhs);
    lhs = rhs;
    rhs = temporary;
}

//! @}

}}
