﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Device Tree のレジスタ空間関連の機能を提供する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/dt/dt_Types.h>

#include <nn/dt/detail/dt_RegisterApi.h>

namespace nn { namespace dt {

//! @name レジスタ空間関連の機能を提供する API
//! @{

/**
 * @brief       指定されたノードが扱うレジスタ空間のアドレスを取得します。
 *
 * @param[out]  pOutAddress   レジスタ空間のアドレス
 * @param[in]   pNode         ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyTypeError}
 *   @handleresult{nn::dt::ResultSizeIncorrect}
 * @endretresult
 *
 * @details     pNode に指定したノードのレジスタアドレス (reg プロパティの前半部) を pOutAddress に格納します。
 *              #address-cells プロパティで指定される Device Tree 上のレジスタアドレスのサイズと T のサイズが一致しなかった場合には、
 *              ResultPropertyTypeError を返します。
 *              reg プロパティに複数のレジスタが定義されていた場合には、ResultSizeIncorrect を返します。
 *              複数のレジスタが定義されている場合は @ref nn::dt::GetRegisterAddressArray() や @ref nn::dt::GetRegisterAddressList() を利用してください。
 *
 */
template <typename T>
nn::Result GetRegisterAddress(T* pOutAddress, const Node* pNode) NN_NOEXCEPT
{
    return detail::GetRegisterAddress(reinterpret_cast<char*>(pOutAddress), pNode, sizeof(T));
}

/**
 * @brief       指定されたノードが扱うレジスタ空間のアドレスの配列を取得します。
 *
 * @param[out]  pOutArray レジスタ空間のアドレスの配列
 * @param[in]   count     配列の長さ
 * @param[in]   pNode     ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyTypeError}
 *   @handleresult{nn::dt::ResultSizeIncorrect}
 * @endretresult
 *
 * @details     pNode に指定したノードのレジスタアドレスの配列を pOutAddress に格納します。
 *              #address-cells プロパティで指定される Device Tree 上のレジスタアドレスのサイズと T のサイズが一致しなかった場合には、
 *              ResultPropertyTypeError を返します。
 *              reg プロパティ中のアドレス・サイズのペアの数が count と一致しなかった場合には、
 *              ResultSizeIncorrect を返します。
 *              count との一致比較を行いたくない場合は、@ref nn::dt::GetRegisterAddressList() を使用してください。
 *
 */
template <typename T>
nn::Result GetRegisterAddressArray(T* pOutArray, int count, const Node* pNode) NN_NOEXCEPT
{
    return detail::GetRegisterAddressArray(reinterpret_cast<char*>(pOutArray), count, pNode, sizeof(T));
}

/**
 * @brief       指定されたノードが扱うレジスタ空間のアドレスの配列を取得します。
 *
 * @param[out]  pOutList  レジスタ空間のアドレスの配列
 * @param[in]   pOutCount 配列の長さ
 * @param[in]   maxCount  pOutList に格納できるノードの最大数
 * @param[in]   pNode     ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyTypeError}
 * @endretresult
 *
 * @details     基本的な仕様は @ref nn::dt::GetRegisterAddressArray() と同等ですが、配列長の一致比較を行わない点が異なります。
 *              reg プロパティ中のアドレス・サイズのペアの数が maxCount と一致しなかった場合には、
 *              可能なだけ pOutList にレジスタアドレスをコピーします。
 *
 */
template <typename T>
nn::Result GetRegisterAddressList(T* pOutList, int* pOutCount, int maxCount, const Node* pNode) NN_NOEXCEPT
{
    return detail::GetRegisterAddressList(reinterpret_cast<char*>(pOutList), pOutCount, maxCount, pNode, sizeof(T));
}

/**
 * @brief       指定されたノードが扱うレジスタ空間のサイズを取得します。
 *
 * @param[out]  pOutSize      レジスタ空間のサイズ
 * @param[in]   pNode         ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyTypeError}
 *   @handleresult{nn::dt::ResultSizeIncorrect}
 * @endretresult
 *
 * @details     pNode に指定したノードのレジスタサイズ (reg プロパティ後半部) を pOutSize に格納します。
 *              #size-cells プロパティで指定される Device Tree 上のレジスタサイズのサイズと T のサイズが一致しなかった場合には、
 *              ResultPropertyTypeError を返します。
 *              reg プロパティに複数のレジスタが定義されていた場合には、ResultSizeIncorrect を返します。
 *              複数のレジスタが定義されている場合は @ref nn::dt::GetRegisterSizeArray() や @ref nn::dt::GetRegisterSizeList() を利用してください。
 *
 */
template <typename T>
nn::Result GetRegisterSize(T* pOutSize, const Node* pNode) NN_NOEXCEPT
{
    return detail::GetRegisterSize(reinterpret_cast<char*>(pOutSize), pNode, sizeof(T));
}

/**
 * @brief       指定されたノードが扱うレジスタ空間のサイズの配列を取得します。
 *
 * @param[out]  pOutArray レジスタ空間のサイズの配列
 * @param[in]   count     配列の長さ
 * @param[in]   pNode     ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyTypeError}
 *   @handleresult{nn::dt::ResultSizeIncorrect}
 * @endretresult
 *
 * @details     pNode に指定したノードのレジスタサイズの配列を pOutAddress に格納します。
 *              #size-cells プロパティで指定される Device Tree 上のレジスタサイズのサイズと T のサイズが一致しなかった場合には、
 *              ResultPropertyTypeError を返します。
 *              reg プロパティ中のアドレス・サイズのペアの数が count と一致しなかった場合には、
 *              ResultSizeIncorrect を返します。
 *              count との一致比較を行いたくない場合は、@ref nn::dt::GetRegisterSizeList() を使用してください。
 *
 */
template <typename T>
nn::Result GetRegisterSizeArray(T* pOutArray, int count, const Node* pNode) NN_NOEXCEPT
{
    return detail::GetRegisterSizeArray(reinterpret_cast<char*>(pOutArray), count, pNode, sizeof(T));
}

/**
 * @brief       指定されたノードが扱うレジスタ空間のサイズの配列を取得します。
 *
 * @param[out]  pOutList  レジスタ空間のサイズの配列
 * @param[in]   pOutCount 配列の長さ
 * @param[in]   maxCount  pOutArray に格納できるノードの最大数
 * @param[in]   pNode     ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultPropertyTypeError}
 * @endretresult
 *
 * @details     基本的な仕様は @ref nn::dt::GetRegisterSizeArray() と同等ですが、配列長の一致比較を行わない点が異なります。
 *              reg プロパティ中のアドレス・サイズのペアの数が maxCount と一致しなかった場合には、
 *              可能なだけ pOutList にレジスタアドレスをコピーします。
 *
 */
template <typename T>
nn::Result GetRegisterSizeList(T* pOutList, int* pOutCount, int maxCount, const Node* pNode) NN_NOEXCEPT
{
    return detail::GetRegisterSizeList(reinterpret_cast<char*>(pOutList), pOutCount, maxCount, pNode, sizeof(T));
}

//! @}

}}
