﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   デバイスドライバサブシステムフレームワークに関する公開ヘッダファイル
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ddsf/ddsf_CastSafe.h>
#include <nn/ddsf/detail/ddsf_ForEach.h>
#include <nn/ddsf/detail/ddsf_Log.h>

namespace nn { namespace ddsf {

//! @name デバイスドライバサブシステムフレームワーク関連 API
//! @{

/**
* @brief    デバイスのアクセスモードを表す型です。
*/
enum AccessMode
{
    AccessMode_None = 0x0, //!< いかなる権限も持たない無効なアクセスモードです。通常使用しません。
    AccessMode_Read = 0x1 << 0, //!< アクセス対象の状態の取得のみを行うアクセスモードです。
    AccessMode_Write = 0x1 << 1, //!< アクセス対象の状態への変更を行うアクセスモードです。
    AccessMode_ReadAndWrite = (AccessMode_Read | AccessMode_Write) //!< アクセス対象の状態の取得および変更の両方を行うアクセスモードです。
};

class ISession;
class IDevice;

/**
* @brief        デバイスに対するセッションをオープンします。
*
* @param[in,out]    pDevice     デバイス
* @param[in,out]    pSession    セッション
* @param[in]        accessMode  アクセスモード
*
* @retresult
*       @handleresult{nn::ddsf::ResultAccessModeDenied, 指定したアクセスモードではオープンできません。}
* @endretresult
*
* @pre
*   - pDevice != nullptr
*   - pSession != nullptr
*   - pSession->IsOpen() == false
*
* @post
*   - pSession->IsOpen() == true
*   - pDevice->HasAnyOpenSession() == true
*
* @detail
*   デバイスに対するセッションをオープンし、セッションとデバイスの間の関連づけを行います。 @n
*   関連づけとは、以下の二つの操作を指します。
*
*   - セッションオブジェクトにデバイスをアタッチする
*   - デバイスオブジェクトの持つセッションリストにセッションを追加する
*/
nn::Result OpenSession(IDevice* pDevice, ISession* pSession, AccessMode accessMode) NN_NOEXCEPT;

/**
* @brief        セッションをクローズし、デバイスとの関連づけを解除します。
*
* @param[in,out]    pSession    セッション
*
* @pre
*   - pSession != nullptr
*
* @post
*   - pSession->IsOpen() == false
*
* @detail
*   セッションをクローズし、デバイスとの関連づけを解除します。
*/
void CloseSession(ISession* pSession) NN_NOEXCEPT;

/**
* @brief        ドライバサブシステムにおけるセッションを表す抽象クラスです。
*
* @detail
*   ドライバサブシステムにおけるセッションを表す抽象クラスです。@n
*   このクラスを直接実体化することはできません。@n
*   ドライバサブシステム実装時は、このクラスを public 継承してサブシステム固有のセッションオブジェクトのクラスを実装してください。 @n
*   @n
*   本クラスから派生クラスへの安全なダウンキャストには SafeCastTo<>(), SafeCastToPointer<>() メソッドを使用することを強く推奨します。 @n
*   安全なダウンキャストを使用するには、派生クラスで @ref NN_DDSF_CAST_SAFE_DECL マクロや @ref NN_DDSF_CAST_SAFE_DEFINE マクロを使用した型情報タグの定義が必要です。 @n
*   詳細は @ref ICastSafe のリファレンスを参照してください。
*/
class ISession : public ICastSafe
{
    friend nn::Result OpenSession(IDevice*, ISession*, AccessMode) NN_NOEXCEPT;
    friend void CloseSession(ISession*) NN_NOEXCEPT;

    NN_DDSF_CAST_SAFE_DECL;

private:
    nn::util::IntrusiveListNode m_ListNode;
    friend class nn::util::IntrusiveListMemberNodeTraits<ISession, &ISession::m_ListNode>;

public:
    typedef nn::util::IntrusiveList<
        ISession, nn::util::IntrusiveListMemberNodeTraits<ISession, &ISession::m_ListNode>
    > List;

public:
    //! @name 生成と破棄
    //! @{

    /**
    * @brief    コンストラクタです。
    */
    ISession() NN_NOEXCEPT
    {
    }

protected:
    /**
    * @brief    デストラクタです。
    *
    * @detail
    *   デストラクタです。 @n
    *   本クラスは継承を前提としており、直接の実体化や基底クラスに対する delete を禁止するため protected non-virtual としています。
    */
    ~ISession() NN_NOEXCEPT
    {
        DetachDevice();
        NN_SDK_ASSERT(!IsOpen());
    }

    //! @}

public:

    //! @name セッションとデバイスの関連づけ
    //! @{

    /**
    * @brief    本オブジェクトを渡されたリストに追加します。
    *
    * @param[in,out]    pList   操作対象のリスト
    *
    * @pre
    *   - pList != nullptr
    */
    void AddTo(List* pList) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pList);
        pList->push_back(*this);
    }

    /**
    * @brief    本オブジェクトを渡されたリストから削除します。
    *
    * @param[in,out]    pList   操作対象のリスト
    *
    * @pre
    *   - pList != nullptr
    */
    void RemoveFrom(List* pList) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pList);
        pList->erase(pList->iterator_to(*this));
    }

    /**
    * @brief    本オブジェクトがリストに追加された状態かどうかを返します。
    *
    * @return   リストに追加された状態かどうか
    */
    bool IsLinkedToList() const NN_NOEXCEPT
    {
        return m_ListNode.IsLinked();
    }

    /**
    * @brief    セッションがオープン状態である、すなわちデバイスがアタッチされているかどうかを返します。
    *
    * @return   セッションがオープン状態であるかどうか
    */
    bool IsOpen() const NN_NOEXCEPT
    {
        return (m_pDevice != nullptr);
    }

    /**
    * @brief    セッションにアタッチされているデバイスへの参照を取得します。
    *
    * @pre
    *   - IsOpen() == true
    *
    * @return   デバイスへの参照
    */
    IDevice& GetDevice() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsOpen());
        return *m_pDevice;
    }

    /**
    * @brief    セッションにアタッチされているデバイスへの const 参照を取得します。
    *
    * @pre
    *   - IsOpen() == true
    *
    * @return   デバイスへの const 参照
    */
    const IDevice& GetDevice() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsOpen());
        return *m_pDevice;
    }

    //! @}

    //! @name アクセス権限
    //! @{

    /**
    * @brief    要求されているアクセスが、セッションのアクセスモードの範疇に含まれるかを判定します。
    *
    * @pre
    *   - IsOpen() == true
    *
    * @return   要求されているアクセスがセッションのアクセスモードの範疇に含まれるかどうか
    */
    bool CheckAccess(AccessMode requestedAccess) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsOpen());
        return (~m_AccessMode & requestedAccess) == 0;
    }

    //! @}

private:
    //! @name セッションとデバイスの関連づけ（プライベート）
    //! @{

    // 以下は OpenSession/CloseSession メソッドの内部からのみアクセスされる

    /**
    * @brief    セッションにデバイスをアタッチします。
    *
    * @param[in]    pDevice     アタッチするデバイス
    * @param[in]    accessMode  デバイスへのアクセスモード
    *
    * @pre
    *   - pDevice != nullptr
    *
    * @post
    *   - IsOpen() == true
    */
    void AttachDevice(IDevice* pDevice, AccessMode accessMode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pDevice);
        m_pDevice = pDevice;
        m_AccessMode = accessMode;
        NN_SDK_ASSERT(IsOpen());
    }

    /**
    * @brief    セッションに関連づいているデバイスをデタッチします。
    *
    * @post
    *   - IsOpen() == false
    */
    void DetachDevice() NN_NOEXCEPT
    {
        m_pDevice = nullptr;
        m_AccessMode = AccessMode_None;
        NN_SDK_ASSERT(!IsOpen());
    }

    //! @}

private:
    IDevice* m_pDevice{ nullptr }; //!< 関連付けられたデバイス
    AccessMode m_AccessMode{ AccessMode_None }; //!< セッションのアクセスモード
};

//! @}

}} // namespace nn::ddsf
