﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief コールバック登録式のイベントハンドラに関する宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_SdkConditionVariable.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/ddsf/ddsf_IEventHandler.h>

namespace nn { namespace ddsf {

//! @name イベントハンドラ関連 API
//! @{

/**
* @brief   イベントハンドラルーパです。
*
* @detail
*   イベントハンドラルーパは、登録されたイベントハンドラのイベントを同時待ちし、シグナルされた際に対応するハンドラを呼び出します。 @n
*   本クラス専用のスレッドから LoopAuto() メソッドを呼び出して実行してください。
*/
class EventHandlerLooper
{
    NN_DISALLOW_COPY(EventHandlerLooper);
    NN_DISALLOW_MOVE(EventHandlerLooper);

public:
    /**
    * @brief デフォルトコンストラクタです。
    */
    EventHandlerLooper() NN_NOEXCEPT;

    /**
    * @brief デストラクタです。
    */
    ~EventHandlerLooper() NN_NOEXCEPT;

    /**
    * @brief イベントハンドラルーパを初期化します。
    *
    * @pre
    *   - イベントハンドラルーパが未初期化状態である
    *
    * @post
    *   - イベントハンドラルーパが初期化状態である
    *
    * @detail
    *   イベントハンドラルーパを初期化します。
    */
    void Initialize() NN_NOEXCEPT;

    /**
    * @brief イベントハンドラルーパを終了します。
    *
    * @pre
    *   - イベントハンドラルーパが初期化状態である
    *   - イベントハンドラルーパが未実行状態である
    *
    * @post
    *   - イベントハンドラルーパが未初期化状態である
    *
    * @detail
    *   イベントハンドラルーパを終了します。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief イベントハンドラルーパにハンドラを登録します。
    *
    * @param[in]    pHandler    ハンドラ
    *
    * @pre
    *   - イベントハンドラルーパが初期化状態である
    *   - pHandler != nullptr;
    *   - pHandler の IEventHandler::IsRegistered() == false
    *
    * @post
    *   - pHandler の IEventHandler::IsRegistered() == true
    *
    * @detail
    *   イベントハンドラルーパにハンドラを登録します。 @n
    *   イベントハンドラルーパが実行状態である場合、多重待ちを行っているスレッドと同期して、多重待ちを一時的に停止した状態で処理を行います。 @n
    *   この関数を多重待ちを行っているスレッド上で呼び出した場合、すなわち登録済のハンドラ内の HandleEvent() から呼び出した場合は、 @n
    *   呼び出したコンテキスト上で登録処理を行います。
    */
    void RegisterHandler(IEventHandler* pHandler) NN_NOEXCEPT;

    /**
    * @brief イベントハンドラルーパからハンドラの登録を解除します。
    *
    * @param[in]    pHandler    ハンドラ
    *
    * @pre
    *   - イベントハンドラルーパが初期化状態である
    *   - pHandler != nullptr;
    *   - pHandler の IEventHandler::IsRegistered() == true
    *
    * @post
    *   - pHandler の IEventHandler::IsRegistered() == false
    *
    * @detail
    *   イベントハンドラルーパからハンドラの登録を解除します。 @n
    *   イベントハンドラルーパが実行状態である場合、多重待ちを行っているスレッドと同期して、多重待ちを一時的に停止した状態で処理を行います。 @n
    *   この関数を多重待ちを行っているスレッド上で呼び出した場合、すなわち登録済のハンドラ内の HandleEvent() から呼び出した場合は、 @n
    *   呼び出したコンテキスト上で解除処理を行います。
    */
    void UnregisterHandler(IEventHandler* pHandler) NN_NOEXCEPT;

    /**
    * @brief イベントハンドラルーパからハンドラの登録を解除します。
    *
    * @pre
    *   - イベントハンドラルーパが初期化状態である
    *   - イベントハンドラルーパが未実行状態である
    *
    * @post
    *   - イベントハンドラルーパが未実行状態である
    *
    * @detail
    *   イベントハンドラルーパを実行します。 @n
    *   本関数の呼び出しは、別スレッドから RequestStop() が呼ばれるまで返りません。
    */
    void LoopAuto() NN_NOEXCEPT;

    /**
    * @brief イベントハンドラルーパが実行状態かどうかを返します。
    *
    * @detail
    *   イベントハンドラルーパが実行状態かどうか、すなわち現在 LoopAuto() を呼び出してイベントを待機しているスレッドがいるかどうかを返します。
    */
    bool IsLooping() const NN_NOEXCEPT
    {
        return m_IsLooping;
    }

    /**
    * @brief イベントハンドラルーパが実行状態に入るまで待機します。
    *
    * @detail
    *   イベントハンドラルーパが実行状態に入ったことを安全に待機します。 @n
    *   この関数が返った直後は IsLooping() が true を返すことが保証されます。 @n
    *   別スレッドで LoopAuto() を呼び出させるシーケンスで、 LoopAuto() に入ったことを確実にする必要があるときに使用してください。
    */
    void WaitLoopEnter() NN_NOEXCEPT;

    /**
    * @brief イベントハンドラルーパが実行状態を終えるのを待機します。
    *
    * @detail
    *   イベントハンドラルーパが実行状態を終えるのを待機します。 @n
    *   この関数が返った直後は IsLooping() が false を返すことが保証されます。 @n
    *   RequestStop() 呼び出し後、別スレッドで実行されていた LoopAuto() が返ったことを確実にする必要があるときに使用してください。
    */
    void WaitLoopExit() NN_NOEXCEPT;

    /**
    * @brief イベントハンドラルーパの実行を停止します。
    *
    * @pre
    *   - イベントハンドラルーパが実行状態である
    *   - イベントハンドラのコールバックコンテキスト以外からの呼び出し
    *
    * @post
    *   - イベントハンドラルーパがイベントの待機を停止している
    *
    * @detail
    *   イベントハンドラルーパの実行の停止を要求します。 @n
    *   この関数を呼び出すと、別のスレッドで呼び出されている LoopAuto() から処理が返ります。 @n
    *   登録済のハンドラの HandleEvent() 内から本関数を呼び出さないでください。
    */
    void RequestStop() NN_NOEXCEPT;

    /**
    * @brief 現在のスレッドが、イベントハンドラのコールバックコンテキスト上で実行中かどうかを返します。
    *
    * @return   現在のスレッドが、イベントハンドラのコールバックコンテキスト上で実行中か
    *
    * @detail
    *   現在のスレッドが、イベントハンドラのコールバックコンテキスト上で実行中かどうかを返します。 @n
    *   イベントハンドラルーパが実行状態で、登録済のハンドラの HandleEvent() 内から本関数を呼び出すと true が返ります。
    */
    bool IsRunningOnEventHandlerContext() const NN_NOEXCEPT;

    //! @}

private:
    struct LooperControlCommandParam;

private:
    void ConfigureMultiWait(LooperControlCommandParam* pParam) NN_NOEXCEPT;
    void ConfigureMultiWaitBody(LooperControlCommandParam* pParam) NN_NOEXCEPT;

private:
    bool m_IsInitialized{ false };

    // AutoLoop() が実行中の状態かどうかのフラグ、それに対応する条件変数（ロックには m_LooperControlLock を使用）
    bool m_IsLooping{ false };
    nn::os::SdkConditionVariable m_IsLoopingCondition;

    // イベントを受けるスレッド本体と、それに使う多重待ちオブジェクト
    nn::os::MultiWaitType m_Waiter;

    // LoopAuto() を呼んでいるスレッドコンテキストからの API 呼び出しかどうかを判別するためのポインタ
    nn::os::ThreadType* m_pLooperThread{ nullptr };

    // ルーパースレッド制御コマンド用のメンバ群
    nn::os::Event m_LooperControlEvent{ nn::os::EventClearMode_AutoClear };
    nn::os::MultiWaitHolderType m_LooperControlEventHolder;
    LooperControlCommandParam* m_pLooperControlCommandParam{ nullptr };
    nn::os::LightEvent m_LooperControlCommandDoneEvent{ nn::os::EventClearMode_AutoClear };
    nn::os::SdkMutex m_LooperControlLock;

};

//! @}

}} // nn::ddsf
