﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief デバイスコードエントリに関する宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_DeviceCode.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_IntrusiveList.h>

#include <nn/ddsf/ddsf_DeviceCodeEntry.h>
#include <nn/ddsf/detail/ddsf_ForEach.h>

namespace nn { namespace ddsf {

class IDevice;

//! @name デバイスコードエントリ関連 API
//! @{

/**
* @brief    デバイスコードエントリのリソース管理を行うユーティリティクラスです。
*
* @detail
*   デバイスコードエントリのリソース管理を行うユーティリティクラスです。 @n
*   デバイスコードエントリのリソースバッファは、構築時にコンストラクタに渡す必要があります。
*/
class DeviceCodeEntryManager
{
public:
    /**
    * @brief    コンストラクタです。
    *
    * @tparam   EntryHolderContainerT     管理対象のデバイスコードエントリのリソースを与えるコンテナ型です。
    *                                     DeviceCodeEntryHolder 型を要素として返す C++ の Range-based for loop の range-expression として使用できる必要があります。
    *
    * @param[in]    pContainer  管理対象のデバイスコードエントリのコンテナです。
    *
    * @pre
    *   - pContainer != nullptr
    *
    * @detail
    *   コンストラクタです。 @n
    *   渡されたコンテナの全要素をイテレートし、内部のフリーリストに追加します。
    */
    template<typename EntryHolderContainerT>
    explicit DeviceCodeEntryManager(EntryHolderContainerT* pContainer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContainer);
        m_Capacity = 0;
        m_FreeEntryList.clear();
        for ( auto&& entry : *pContainer )
        {
            m_FreeEntryList.push_back(entry);
            ++m_Capacity;
        }

        m_UsedEntryList.clear();
    }

    /**
    * @brief    デストラクタです。
    */
    ~DeviceCodeEntryManager() NN_NOEXCEPT
    {
        m_FreeEntryList.clear();
        m_UsedEntryList.clear();
        m_Capacity = 0;
    }

    /**
    * @brief    管理可能なデバイスコードエントリの総数を返します。
    *
    * @return   管理可能なデバイスコードエントリの総数
    *
    * @detail
    *   管理可能なデバイスコードエントリの総数を返します。
    */
    int GetCapacity() const NN_NOEXCEPT
    {
        return m_Capacity;
    }

    /**
    * @brief    すべてのデバイスコードエントリを削除し、管理情報をリセットします。
    *
    * @detail
    *   すべてのデバイスコードエントリを削除し、管理情報をリセットします。
    */
    void Reset() NN_NOEXCEPT;

    /**
    * @brief    デバイスコードエントリを追加します。
    *
    * @param[in]    deviceCode  デバイスコード
    * @param[in]    pDevice     デバイスオブジェクト
    *
    * @retresult
    *   @handleresult{ResultSuccess}
    *   @handleresult{ResultOutOfResource, デバイスコードエントリオブジェクトの空きがありません。}
    * @endretresult
    *
    * @pre
    *   - deviceCode が未登録
    *   - pDevice != nullptr
    *   - pDevice->IsDriverAttached() == true
    *
    * @detail
    *   デバイスコードエントリを追加します。
    */
    nn::Result Add(nn::DeviceCode deviceCode, IDevice* pDevice) NN_NOEXCEPT;

    /**
    * @brief    デバイスコードエントリを削除します。
    *
    * @param[in]    deviceCode  デバイスコード
    *
    * @return   該当するデバイスコードが存在し、削除が行われた場合に true を返します。該当デバイスコードがすでに存在しなかった場合 false を返します。
    *
    * @detail
    *   指定したデバイスコードを持つデバイスコードエントリを削除します。
    */
    bool Remove(nn::DeviceCode deviceCode) NN_NOEXCEPT;

    /**
    * @brief 有効なデバイスコードエントリの全てに対して指定したメソッドを呼び出します。
    *
    * @tparam   FuncT   DeviceCodeEntryHolder* を引数に取り bool 評価可能な値を返す関数として振る舞える型
    *
    * @param[in] func   呼び出すメソッド
    *
    * @return   func が true を返した回数
    *
    * @detail
    *   有効なデバイスコードエントリの全てに対して指定したメソッドを呼び出します。 @n
    *   呼び出された @a func が true を返す限り、続く要素へのメソッド呼び出しを継続します。 @n
    *   @a func が false を返した時点で以降のメソッド呼び出しを中止します。 @n
    *   @n
    *   本関数の実行は、内部でデバイスコードエントリの追加操作と排他制御されます。
    */
    template<typename FuncT>
    int ForEachEntry(FuncT func) NN_NOEXCEPT
    {
        return detail::ForEach(
            &m_UsedEntryList,
            [&](DeviceCodeEntryHolder* pEntryHolder) NN_NOEXCEPT -> bool
            {
                NN_SDK_ASSERT_NOT_NULL(pEntryHolder);
                NN_SDK_ASSERT(pEntryHolder->IsBuilt());
                return func(&pEntryHolder->Get());
            },
            &m_EntryListLock);
    }

    /**
    * @brief 有効なデバイスコードエントリの全てに対して指定したメソッドを呼び出します。 (const)
    *
    * @tparam   FuncT   const DeviceCodeEntryHolder* を引数に取り bool 評価可能な値を返す関数として振る舞える型
    *
    * @param[in] func   呼び出すメソッド
    *
    * @return   func が true を返した回数
    *
    * @detail
    *   呼び出すメソッドの引数が const 参照になる以外は非 const 版と同じです。 @n
    *   const メンバ関数内から呼び出すことができます。
    */
    template<typename FuncT>
    int ForEachEntry(FuncT func) const NN_NOEXCEPT
    {
        return detail::ForEach(
            &m_UsedEntryList,
            [&](const DeviceCodeEntryHolder* pEntryHolder) NN_NOEXCEPT -> bool
            {
                NN_SDK_ASSERT_NOT_NULL(pEntryHolder);
                NN_SDK_ASSERT(pEntryHolder->IsBuilt());
                return func(&pEntryHolder->Get());
            },
            &m_EntryListLock);
    }

    /**
    * @brief 指定したデバイスコードに対応するデバイスコードエントリを取得します。
    *
    * @param[out]   ppOutDeviceCodeEntry    該当するデバイスコードエントリへのポインタを格納するポインタ
    * @param[in]    code                    デバイスコード
    *
    * @retresult
    *   @handleresult{ResultSuccess}
    *   @handleresult{ResultInvalidArgument}
    *   @handleresult{ResultDeviceCodeNotFound}
    * @endretresult
    *
    * @pre
    *   - ppOutDeviceCodeEntry != nullptr
    *
    * @detail
    *   指定したデバイスコードに対応するデバイスコードエントリを取得します。
    */
    nn::Result FindDeviceCodeEntry(DeviceCodeEntry** ppOutDeviceCodeEntry, nn::DeviceCode code) NN_NOEXCEPT;

    /**
    * @brief 指定したデバイスコードに対応するデバイスコードエントリを取得します。 (const)
    *
    * @param[out]   ppOutDeviceCodeEntry    該当するデバイスコードエントリへのポインタを格納するポインタ
    * @param[in]    code                    デバイスコード
    *
    * @retresult
    *   @handleresult{ResultSuccess}
    *   @handleresult{ResultInvalidArgument}
    *   @handleresult{ResultDeviceCodeNotFound}
    * @endretresult
    *
    * @pre
    *   - ppOutDeviceCodeEntry != nullptr
    *
    * @detail
    *   指定したデバイスコードに対応するデバイスコードエントリを取得します。 @n
    *   const メンバ関数内から呼び出すことができます。
    */
    nn::Result FindDeviceCodeEntry(const DeviceCodeEntry** ppOutDeviceCodeEntry, nn::DeviceCode code) const NN_NOEXCEPT;

    /**
    * @brief 指定したデバイスコードに対応するデバイスオブジェクトを取得します。
    *
    * @param[out]   ppOutDevice             該当するデバイスオブジェクトへのポインタを格納するポインタ
    * @param[in]    code                    デバイスコード
    *
    * @retresult
    *   @handleresult{ResultSuccess}
    *   @handleresult{ResultInvalidArgument}
    *   @handleresult{ResultDeviceCodeNotFound}
    * @endretresult
    *
    * @pre
    *   - ppOutDevice != nullptr
    *
    * @detail
    *   指定したデバイスコードに対応するデバイスオブジェクトを取得します。
    */
    nn::Result FindDevice(IDevice** ppOutDevice, nn::DeviceCode code) NN_NOEXCEPT;

    /**
    * @brief 指定したデバイスコードに対応するデバイスオブジェクトを取得します。 (const)
    *
    * @param[out]   ppOutDevice             該当するデバイスオブジェクトへのポインタを格納するポインタ
    * @param[in]    code                    デバイスコード
    *
    * @retresult
    *   @handleresult{ResultSuccess}
    *   @handleresult{ResultInvalidArgument}
    *   @handleresult{ResultDeviceCodeNotFound}
    * @endretresult
    *
    * @pre
    *   - ppOutDevice != nullptr
    *
    * @detail
    *   指定したデバイスコードに対応するデバイスオブジェクトを取得します。 @n
    *   const メンバ関数内から呼び出すことができます。
    */
    nn::Result FindDevice(const IDevice** ppOutDevice, nn::DeviceCode code) const NN_NOEXCEPT;

private:
    nn::ddsf::DeviceCodeEntryHolder::List m_UsedEntryList;
    nn::ddsf::DeviceCodeEntryHolder::List m_FreeEntryList;
    mutable os::SdkMutex m_EntryListLock;
    int m_Capacity;
};

//! @}

}} // nn::ddsf
