﻿using Nintendo.Nact;
using Nintendo.Nact.Utilities;
using Nintendo.Nact.Utilities.ProgramExecution;
using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Threading;
using static Nintendo.Nact.Extensions.ProgramExecuterExtensions;

namespace SigloNact.Utilities
{
    public class GitExecuter
    {
        private readonly IProgramExecuter _programExecuter;
        private readonly IDictionary<string, string> _environmentVariables;

        public GitExecuter(
            IProgramExecuter programExecuter,
            IDictionary<string, string> environmentVariables)
        {
            this._programExecuter = programExecuter;
            this._environmentVariables = environmentVariables;
        }

        public IEnumerable<string> GitLsExrepoFiles(string rootPath)
        {
            return GetGitResultsLines(rootPath, @"exrepo list-tree");
        }

        public IEnumerable<string> GetGitResultsLines(string rootPath, string arguments)
        {
            var parameters = new ProgramExecutionParameters
            {
                FileName = Path.Combine(GetGitInstallPath(), "cmd", "git.exe"),
                WorkingDirectory = rootPath,
                Arguments = @"-c core.quotepath=true " + arguments,
                EnvironmentVariables = _environmentVariables,
            };

            var programResult = _programExecuter.ExecuteForResult(parameters, CancellationToken.None);
            using (var sr = new StreamReader(new MemoryStream(programResult.StandardOutput)))
            {
                var ret = new List<string>();
                while (true)
                {
                    var line = sr.ReadLine();
                    if (line == null)
                    {
                        break;
                    }
                    ret.Add(Unquote(line));
                }
                return ret;
            }
        }

        private string GetGitInstallPath()
        {
            var s = Util.GetValueOrDefault(_environmentVariables, "SIGLO_GIT_PATH")
                ?? RegistryUtil.GetSoftwareValue(@"Microsoft\Windows\CurrentVersion\Uninstall\Git_is1", "InstallLocation") as string;
            if (s == null)
            {
                throw new ErrorException("The SIGLO_GIT_PATH environment variable is not defined; Git installation path cannot be obtained. Note: SIGLO_GIT_PATH is defined by siglo.cmd.");
            }
            return s;
        }

        private static string Unquote(string s)
        {
            if (s.StartsWith("\"", StringComparison.Ordinal))
            {
                // '"' から始まる場合エスケープされているため処理が必要
                var list = new List<byte>();
                var i = 1;
                while (i < s.Length - 1)
                {
                    if (s[i] == '\\')
                    {
                        // \ で始まる場合には \ に続けて 3 ケタの 8 進数が続く
                        list.Add((byte)Convert.ToInt32(s.Substring(i + 1, 3), 8));
                        i += 4;
                    }
                    else
                    {
                        // \ で始まらない場合には ASCII 文字の範囲として読める
                        list.Add((byte)s[i]);
                        i += 1;
                    }
                }
                return Encoding.UTF8.GetString(list.ToArray());
            }
            else
            {
                return s;
            }
        }
    }
}
