﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.FileSystem;
using SigloNact.Build;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using static Nintendo.Nact.Extensions.FormattableStringExtensions;
using static System.FormattableString;

namespace SigloNact.BuiltIns.ToolChain.GccClang
{
    public static class GccClangUtil
    {
        private static readonly Regex ErrorDetectorRegex = new Regex(": (error|fatal error):");
        private static readonly Regex WarningDetectorRegex = new Regex(": warning:");

        public static void OutputModifier(List<string> text)
        {
        }

        public static IEnumerable<string> ErrorDetector(IReadOnlyList<string> lines)
        {
            return lines.Where(x => ErrorDetectorRegex.IsMatch(x));
        }

        public static IEnumerable<string> WarningDetector(IReadOnlyList<string> lines)
        {
            return lines.Where(x => WarningDetectorRegex.IsMatch(x));
        }

        private static readonly IReadOnlyCollection<string> CLanguageOptions = new string[] { "-x", "c", };
        private static readonly IReadOnlyCollection<string> CxxLanguageOptions = new string[] { "-x", "c++", };
        private static readonly IReadOnlyCollection<string> AsLanguageOptions = new string[] { "-x", "assembler-with-cpp", };

        public static IReadOnlyCollection<string> GetLanguageOptions(CompileLanguage language)
        {
            switch (language)
            {
                case CompileLanguage.C:
                    return CLanguageOptions;
                case CompileLanguage.Cxx:
                    return CxxLanguageOptions;
                case CompileLanguage.Asm:
                    return AsLanguageOptions;
                default:
                    throw new InvalidOperationException("should never be reached");
            }
        }

        public static IGccClangCommandHelper CreateCommandHelper(GccClangToolChainSpecifier toolChainSpecifier)
        {
            switch (toolChainSpecifier.ToolChainKind)
            {
                case ToolChainKind.Gcc:
                    return new GccCommandHelper(toolChainSpecifier);
                case ToolChainKind.Clang:
                    return new ClangCommandHelper(toolChainSpecifier);
                case ToolChainKind.Msvc:
                    throw new ArgumentException("Msvc not supported here.");
                default:
                    throw new InvalidOperationException("Should never be reached.");
            }
        }

        public static string GetLinkerNameForUseLd(ToolChainKind toolChainKind, LinkerKind linkerKind)
        {
            var postfix = getPostfix();

            switch (linkerKind)
            {
                case LinkerKind.Ld:
                    return "bfd" + postfix;
                case LinkerKind.Gold:
                    return "gold" + postfix;
                case LinkerKind.Lld:
                    return "lld" + postfix;
                default:
                    throw new InvalidOperationException("should never be reached.");
            }

            string getPostfix()
            {
                switch (toolChainKind)
                {
                    case ToolChainKind.Gcc:
                        return "";
                    case ToolChainKind.Clang:
                        // Rynda においては、.exe を含む名前を与える必要がある
                        return ".exe";
                    case ToolChainKind.Msvc:
                        throw new ArgumentException("Msvc not supported here.");
                    default:
                        throw new InvalidOperationException("Should never be reached.");
                }
            }
        }

        // 実際これらは関数で取得してあたえてもらうとよい
        public static void AddArchitectureOptions(List<string> flags, GccClangArchitectureParameters architectureParameters)
        {
            switch (architectureParameters.Cpu)
            {
                case "cortex-a9":
                    flags.Add("-mcpu=cortex-a9");
                    flags.Add(architectureParameters.UseNeon ? "-mfpu=neon" : "-mfpu=vfpv3-fp16");
                    flags.Add("-mfloat-abi=hard");
                    break;
                case "cortex-a15":
                    flags.Add("-mcpu=cortex-a15");
                    flags.Add(architectureParameters.UseNeon ? "-mfpu=neon-vfpv4" : "-mfpu=vfpv4");
                    flags.Add("-mfloat-abi=hard");
                    break;
                case "cortex-a7":
                    flags.Add("-mcpu=cortex-a7");
                    flags.Add(architectureParameters.UseNeon ? "-mfpu=neon-vfpv4" : "-mfpu=vfpv4");
                    flags.Add("-mfloat-abi=hard");
                    break;
                case "cortex-a53-aarch32":
                    flags.Add("-mcpu=cortex-a53");
                    flags.Add("-mfpu=crypto-neon-fp-armv8");
                    flags.Add("-mfloat-abi=hard");
                    break;
                case "cortex-a57-aarch32":
                    flags.Add("-mcpu=cortex-a57");
                    flags.Add("-mfpu=crypto-neon-fp-armv8");
                    flags.Add("-mfloat-abi=hard");
                    break;
                case "cortex-a53-aarch64":
                    flags.Add("-mcpu=cortex-a53+fp+simd+crypto+crc");
                    break;
                case "cortex-a57-aarch64":
                    flags.Add("-mcpu=cortex-a57+fp+simd+crypto+crc");
                    break;
                default:
                    throw new InvalidOperationException(Current($"Unknown Cpu specified: {architectureParameters.Cpu}."));
            }
        }

        public static void AddModelOptions(List<string> flags, ToolChainKind toolChainKind, GccClangArchitectureParameters architectureParameters)
        {
            switch (toolChainKind)
            {
                case ToolChainKind.Gcc:
                case ToolChainKind.Clang:
                    break;
                case ToolChainKind.Msvc:
                    throw new ArgumentException("Msvc not supported here.");
                default:
                    throw new InvalidOperationException("Should never be reached.");
            }

            AddArchitectureOptions(flags, architectureParameters);

            switch (toolChainKind)
            {
                case ToolChainKind.Gcc:
                    switch (architectureParameters.Architecture)
                    {
                        case "arm":
                            flags.Add("-mabi=aapcs-linux");
                            flags.Add("-mthumb-interwork");
                            break;
                        case "aarch64":
                            switch (architectureParameters.Abi)
                            {
                                case "lp64":
                                    flags.Add("-mabi=lp64");
                                    break;
                                case "ilp32":
                                    flags.Add("-mabi=ilp32");
                                    break;
                                default:
                                    throw new InvalidOperationException(Current($"Unknown Cpu specified: {architectureParameters.Abi}."));
                            }
                            break;
                        default:
                            throw new InvalidOperationException(Current($"Unknown architecture specified: {architectureParameters.Architecture}."));
                    }
                    break;
                case ToolChainKind.Clang:
                    switch (architectureParameters.Architecture)
                    {
                        case "arm":
                            flags.Add("-mabi=aapcs-linux");
                            break;
                        case "aarch64":
                            switch (architectureParameters.Abi)
                            {
                                case "lp64":
                                    break;
                                case "ilp32":
                                    flags.Add("-mabi=ilp32");
                                    break;
                                default:
                                    throw new InvalidOperationException(Current($"Unknown Cpu specified: {architectureParameters.Abi}."));
                            }
                            break;
                        default:
                            throw new InvalidOperationException(Current($"Unknown architecture specified: {architectureParameters.Architecture}."));
                    }
                    break;
                case ToolChainKind.Msvc:
                default:
                    throw new InvalidOperationException("Should never be reached.");
            }
        }
    }
}
