﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.FileSystem;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Runtime.Serialization;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using System.Xml.XPath;
using static System.FormattableString;

namespace SigloNact.BuiltIns.Prebuilt
{
    internal static class RequestResolver
    {
        [Serializable]
        public class RequestResolverException : Exception
        {
            public RequestResolverException()
                : base()
            {
            }

            public RequestResolverException(string message)
                : base(message)
            {
            }

            protected RequestResolverException(SerializationInfo info, StreamingContext context)
                : base(info, context)
            {
            }

            public RequestResolverException(string message, Exception innerException)
                : base(message, innerException)
            {
            }
        }

        class BuildResult
        {
            public string BuildId { get; set; }
            public string Revision { get; set; }

            public string CreateArtifactUri(string baseUri)
            {
                return string.Format("{0}/httpAuth/app/rest/builds/id:{1}/artifacts/content", baseUri, BuildId);
            }

            public string CreateWebUrl(string baseUri)
            {
                return string.Format("{0}/viewLog.html?buildId={1}", baseUri, BuildId);
            }
        }

        private static string GetGitInstallPathFromRegistory()
        {
            foreach (var softwareKeyName in new[] { @"SOFTWARE", @"SOFTWARE\Wow6432Node" })
            {
                foreach (var root in new[] { Microsoft.Win32.Registry.LocalMachine, Microsoft.Win32.Registry.CurrentUser })
                {
                    using (var key = root.OpenSubKey(softwareKeyName + @"\Microsoft\Windows\CurrentVersion\Uninstall\Git_is1"))
                    {
                        if (key == null)
                        {
                            continue;
                        }

                        var value = key.GetValue(@"InstallLocation") as string;
                        if (value != null)
                        {
                            return value;
                        }
                    }
                }
            }

            return null;
        }

        private static FilePath GetGitInstallPath()
        {
            var s = Environment.GetEnvironmentVariable("SIGLO_GIT_PATH") ?? GetGitInstallPathFromRegistory();
            if (s == null)
            {
                throw new RequestResolverException("The SIGLO_GIT_PATH environment variable is not defined; Git installation path cannot be obtained. Note: SIGLO_GIT_PATH is defined by siglo.cmd.");
            }
            return FilePath.CreateLocalFileSystemPath(s);
        }

        public static FilePath GetGitCommandPath()
        {
            return GetGitInstallPath().Combine(@"cmd\git.exe");
        }

        public static bool Resolve(
            INactActionHelper helper,
            string baseUri,
            FilePath outputDirectory,
            FilePath baseRevisionFile,
            string searchAncestorCountStr,
            string locator,
            string fields)
        {
            try
            {
                var baseRevision = helper.ReadAllText(baseRevisionFile, Encoding.ASCII);

                Authorization.AuthenticationInfo authenticationInfo;

                try
                {
                    authenticationInfo = Authorization.FindAuthenticationInfo(PrebuiltUtility.GetAuthorizationTargetName());
                }
                catch (Authorization.AuthorizationException e)
                {
                    throw new RequestResolverException(string.Format("Failed to get an account. Please execute 'Integrate/CommandLineTools/CredentialRegister.exe' and register your account."), e);
                }

                helper.Out.Write("================================================================================\n");
                helper.Out.Write("{0}\n", PrebuiltUtility.GetMessageOfAnnouncementOfSkipDownloadPrebuiltResultsOption());
                helper.Out.Write("\n");
                helper.Out.Write("{0}\n", PrebuiltUtility.GetMessageOfAnnouncementOfDetailsPage());
                helper.Out.Write("================================================================================\n");

                var searchAncestorCount = int.Parse(searchAncestorCountStr);
                var searchCount = searchAncestorCount + 1;

                // searchCount の数だけ歴史を遡ってコミットをリストする
                var gitResult = helper.ExecuteProgramForStandardOutput(
                    GetGitCommandPath(),
                    Invariant($"rev-list --first-parent -n {searchCount} {baseRevision}"));

                if (!gitResult.IsSuccessful)
                {
                    return false;
                }

                var revisionList = Encoding.ASCII.GetString(gitResult.StandardOutput).Split();

                // 事前ビルド成果物のジョブ結果のリストを得る
                var buildResultList = CreateBuildResultList(authenticationInfo, baseUri, locator, fields);
                var buildResultLookup = buildResultList.ToLookup(x => x.Revision);

                // コミットをさらい、成果物の存在するコミットを探索します。
                foreach (var revision in revisionList)
                {
                    if (!buildResultLookup.Contains(revision))
                    {
                        continue;
                    }

                    // 成果物の存在するリビジョンを見つけた
                    var buildResult = buildResultLookup[revision].First();

                    if (baseRevision != revision)
                    {
                        helper.Out.Write("================================================================================\n");
                        helper.Out.Write("WARNING: nact-prebuilt-rule donwloads the results which have been built on the revision '{0}'. This revision is older than the revision '{1}' at which the feature branch was created.\n", revision, baseRevision);
                        helper.Out.Write("\n");
                        helper.Out.Write("{0}\n", PrebuiltUtility.GetMessageOfAnnouncementOfDetailsPage());
                        helper.Out.Write("================================================================================\n");
                    }

                    // 事前ビルドのためのファイルを作成する
                    OutputResults(helper, outputDirectory, buildResult, baseUri);

                    return true;
                }

                helper.Out.Write("Could not find an appropriate build result.");
                return false;
            }
            catch (Exception e)
            {
                helper.Out.Write("================================================================================\n");
                helper.Out.Write("error: {0}\n", e.Message);
                helper.Out.Write("{0}\n", PrebuiltUtility.GetMessageOfAnnouncementOfDetailsPage());
                helper.Out.Write("================================================================================\n");
                throw;
            }
        }

        static private IEnumerable<BuildResult> CreateBuildResultList(Authorization.AuthenticationInfo authenticationInfo, string baseUri, string locator, string fields)
        {
            var sourceUri = string.Format("{0}/httpAuth/app/rest/builds?locator={1}&fields={2}", baseUri, locator, fields);

            string response;

            try
            {
                using (var client = new WebClient())
                {
                    client.Credentials = new System.Net.NetworkCredential(authenticationInfo.Username, authenticationInfo.Password);

                    response = client.DownloadString(sourceUri);
                    if (response == string.Empty)
                    {
                        // ダウンロードが空の場合は、認証情報が正しくなかった場合などに起こる。
                        throw new RequestResolverException(PrebuiltUtility.GenerateErrorMessage("Failed to request to a server.", authenticationInfo.Username, authenticationInfo.Password, sourceUri));
                    }
                }
            }
            catch (WebException e)
            {
                throw new RequestResolverException(PrebuiltUtility.GenerateErrorMessage(e.Message, authenticationInfo.Username, authenticationInfo.Password, sourceUri), e);
            }

            var xdocument = XDocument.Parse(response);
            var xbuildList = xdocument.XPathSelectElements("//builds/build");

            return Enumerable.Select(xbuildList, x =>
                new BuildResult
                {
                    BuildId = x.Attribute("id").Value,
                    Revision = x.XPathSelectElement("revisions/revision[@version]").Attribute("version").Value,
                });
        }

        private static void OutputResults(
            INactActionHelper helper,
            FilePath outputDirectory,
            BuildResult buildResult,
            string baseUri)
        {
            helper.WriteAllText(outputDirectory.Combine("Revision"), buildResult.Revision, Encoding.ASCII);

            helper.WriteAllText(outputDirectory.Combine("BuildId"), buildResult.BuildId, Encoding.ASCII);

            helper.WriteAllText(outputDirectory.Combine("ArtifactUri"), buildResult.CreateArtifactUri(baseUri), Encoding.ASCII);

            helper.WriteAllText(outputDirectory.Combine("WebUrl"), buildResult.CreateWebUrl(baseUri), Encoding.ASCII);
        }
    }
}
