﻿using Nintendo.Nact.Actions;
using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Utilities;
using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Linq;
using static Nintendo.Nact.Extensions.ExecutionSideEffectWriterExtensions;
using static Nintendo.Nact.Extensions.NactValueExtensions;

namespace SigloNact.BuiltIns.MSBuild
{
    [NactActionFunctionContainer]
    public static class PreCheckExecuteMsBuildContainer
    {
        [NactActionFunction(Version = 0)]
        public static NactActionResult PreCheckExecuteMsBuild(
            INactActionContext context,
            ExecuteMsBuild executeMsBuild,
            string toolsVersion,
            string vsVersion,
            IReadOnlyCollection<object> sources,
            IReadOnlyCollection<FilePath> allSubRootLibraryBaseDirectories,
            IReadOnlyCollection<string> allowedLibraryBuildTargetNames,
            IReadOnlyCollection<string> allowedToolsVersions,
            IReadOnlyCollection<string> allowedVsVersions)
        {
            var helper = context.Helper;

            // ToolsVersion, VsVersion が許可された範囲かどうか
            if (!allowedToolsVersions.Contains(toolsVersion))
            {
                return NactActionResult.CreateFailure(string.Format(Strings.PreCheckExecuteMsBuild_NotAllowdToolsVersion, toolsVersion, string.Join(",", allowedToolsVersions)), string.Empty);
            }
            if (!allowedVsVersions.Contains(vsVersion))
            {
                return NactActionResult.CreateFailure(string.Format(Strings.PreCheckExecuteMsBuild_NotAllowdVsVersion, vsVersion, string.Join(",", allowedVsVersions)), string.Empty);
            }

            // allowedLibraryBuildTargetNames 以外の nn ライブラリをリンクしていないか
            var badLibs = sources
                .Select(x => x.GetPath())
                .Where(x => !isAllowedLib(x))
                .ToArray();

            if (badLibs.Length != 0)
            {
                var summary = string.Format(Strings.PreCheckExecuteMsBuild_NotAllowedLibrary, badLibs[0]);
                helper.SetErrorSummary(summary);

                foreach (var badlib in badLibs)
                {
                    helper.WriteDiagnosticMessage(
                        DiagnosticsSeverity.Error,
                        string.Format(Strings.PreCheckExecuteMsBuild_NotAllowedLibrary, badlib),
                        null);
                }

                return helper.FinishAsFailure();
            }

            return NactActionResult.Success;

            // allowedLibraryBuildTargetNames に許可されているビルドターゲットのライブラリであれば true を返す。
            // ただし、サブルートのライブラリではない場合は許可されていることにする (true)。
            bool isAllowedLib(FilePath path)
            {
                // 対応する Libraries ディレクトリを得る
                var libraryBaseDirectory = allSubRootLibraryBaseDirectories.FirstOrDefault(x => x.IsAncestorOf(path));
                if (libraryBaseDirectory == null)
                {
                    // サブルートのライブラリではない
                    return true;
                }

                var buildTargetName = libraryBaseDirectory.GetRelativeFilePathTo(path).PathComponents[0].OriginalValue;
                return allowedLibraryBuildTargetNames.Contains(buildTargetName, StringComparer.OrdinalIgnoreCase);
            }
        }
    }
}
