﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Utilities;
using Nintendo.Nact.Values;
using System;
using System.Collections;
using static System.FormattableString;
using System.Collections.Generic;
using Microsoft.Build.Utilities;

namespace SigloNact.BuiltIns.G3dDemo
{
    [NactActionFunctionContainer]
    public static class CreateG3dDemoResourceContainer
    {
        [NactActionFunction]
        public static NactActionResult CreateG3dDemoResource(
            INactActionContext context,
            FilePath path,
            string target,
            string platform,
            string configuration,
            string toolsVersion,
            string vsVersion,
            bool disableFsdbGeneration,
            IReadOnlyCollection<FilePath> jitProfiles = null)
        {
            var helper = context.Helper;

            // .JIT.Profile への書き込みはツール内で制御しているものとしてアクセス追跡結果から除外する
            var result = helper.ExecuteProgram(
                GetMsBuildExePath(toolsVersion, platform),
                MakeMsBuildArguments(path, target, platform, configuration, toolsVersion, vsVersion, disableFsdbGeneration),
                trackFileAccessIgnoredFiles: jitProfiles);
            return helper.FinishWithExitCode(result.ExitCode);
        }

        [NactActionFunction]
        public static NactActionResult CleanG3dDemoResource(
            INactActionContext context,
            FilePath path,
            string platform,
            string configuration,
            string toolsVersion,
            string vsVersion)
        {
            var helper = context.Helper;

            // Clean の場合のみアクセス追跡しない
            var (result, text) = helper.DangerousExecuteProgram(
                GetMsBuildExePath(toolsVersion, platform),
                MakeMsBuildArguments(path, "Clean", platform, configuration, toolsVersion, vsVersion, true),
                new Dictionary<string, string>(),
                NactExecuteProgramFlags.None);
            return helper.FinishWithExitCode(result.ExitCode);
        }

        private static FilePath GetMsBuildExePath(string toolsVersion, string platform)
        {
            var bitness = platform.ToUpperInvariant() == "X64" ?
                DotNetFrameworkArchitecture.Bitness64 : DotNetFrameworkArchitecture.Bitness32;
            return FilePath.CreateLocalFileSystemPath(
                ToolLocationHelper.GetPathToBuildToolsFile("MSBuild.exe", toolsVersion, bitness));
        }

        private static string MakeMsBuildArguments(
            FilePath path,
            string target,
            string platform,
            string configuration,
            string toolsVersion,
            string vsVersion,
            bool disableFsdbGeneration)
        {
            string disableFsdbGenerationString = disableFsdbGeneration ? "True" : "False";
            return string.Join(" ", new string[]
                {
                    Invariant($"\"{path}\""),
                    "/nologo",
                    // 他の MSBuild が生成したノードプロセスを利用しないようにするため、ノードプロセスを使用しないようにする。
                    // MSBuild を Tracker で追跡するとき、ビルド処理がほかのノードプロセスで実行されて追跡できなくなったり、
                    // 他のビルドが追跡しているノードプロセスで実行されて他のビルドを追跡してしまうことを防ぐため。
                    "/maxcpucount:1",
                    "/nodeReuse:false",
                    "/verbosity:Quiet",
                    Invariant($"/toolsversion:{toolsVersion}"),
                    Invariant($"/target:{target}"),
                    // プロパティ
                    $"/property:Configuration=\"{configuration}\"",
                    $"/property:Platform=\"{platform}\"",
                    $"/property:NintendoSdkBuildByInternalTool=true",
                    $"/property:VisualStudioVersion=\"{vsVersion}\"",
                    $"/property:BuildProjectReferences=false",
                    $"/property:UseSharedCompilation=false",
                    $"/property:FsdbGenerationDisabled=\"{disableFsdbGenerationString}\"",
                 });
        }
    }
}
