﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Xml;
using System.Xml.Linq;
using System.Xml.XPath;

namespace SigloNact.BuiltIns.Common
{
    [NactActionFunctionContainer]
    public static class AddOtherXmlContentContainer
    {
        [NactActionFunction]
        public static NactActionResult AddOtherXmlContent(
            INactActionContext context,
            FilePath destinationFilePath,
            FilePath baseSourceFilePath,
            IEnumerable<FilePath> otherSourceFilePaths,
            IEnumerable<string> valueElements,
            IEnumerable<string> containerElements)
        {
            AddOtherXmlContentImpl(
                destinationFilePath.PathString,
                baseSourceFilePath.PathString,
                valueElements,
                containerElements,
                otherSourceFilePaths.Select(x => x.PathString));

            return NactActionResult.CreateSuccess(
                new[] { baseSourceFilePath }.Concat(otherSourceFilePaths).ToArray(),
                new[] { destinationFilePath });
        }

        private enum AddOperationType
        {
            ValueElement,
            ContainerElement,
        }

        private class AddOperation
        {
            public AddOperation(AddOperationType type, string path)
            {
                Type = type;
                Path = path;
            }

            public AddOperationType Type { get; private set; }
            public string Path { get; private set; }
        }

        private static XElement EnsureDescendantElement(XContainer container, IEnumerable<string> descendantElementNames)
        {
            if (!descendantElementNames.Any())
            {
                return (XElement)container;
            }

            var childElementName = descendantElementNames.First();
            var childElement = container.Element(childElementName);
            if (childElement == null)
            {
                childElement = new XElement(childElementName);
                container.Add(childElement);
            }
            return EnsureDescendantElement(childElement, descendantElementNames.Skip(1));
        }

        private static bool HasNonEmptyElement(XContainer xml)
        {
            return xml.Elements().Any(e => e.Value != string.Empty || HasNonEmptyElement(e));
        }

        private static void AddOtherXmlContentImpl(
            XDocument xml,
            IEnumerable<AddOperation> operations,
            string otherSourcePath)
        {
            var otherXml = XDocument.Load(otherSourcePath);

            foreach (var operation in operations)
            {
                var sourceElement = otherXml.XPathSelectElement(operation.Path);
                if (sourceElement == null)
                {
                    continue;
                }
                // 処理した要素は削除する。すべての operations を処理した後に未処理の要素が残ったらエラー
                sourceElement.Remove();

                var descendantElementNames = operation.Path.Split('/');
                var element = EnsureDescendantElement(xml, descendantElementNames);
                switch (operation.Type)
                {
                    case AddOperationType.ValueElement:
                        element.SetValue(sourceElement.Value);
                        break;
                    case AddOperationType.ContainerElement:
                        element.Add(sourceElement.Elements());
                        break;
                    default:
                        break;
                }
            }

            if (HasNonEmptyElement(otherXml))
            {
                throw new ArgumentException(
                    string.Format("{0}: There remains an unconsumed element: {1}", otherSourcePath, otherXml.ToString()),
                    "sourceFilePath");
            }
        }

        /// <summary>
        /// baseSouceFile の XML ファイルに、otherSourceFiles で指定された XML ファイルの内容を追加し、destinationFile に保存します。
        /// 追加方法は、valueElementPaths, containerElementPaths で指定します。
        /// </summary>
        /// <param name="destinationFile">保存先の XML ファイル</param>
        /// <param name="baseSourceFile">追加する元になる XML ファイル</param>
        /// <param name="valueElementPaths">指定された要素について、元になる XML の要素の値を、他の XML ファイルの要素の値で置き換えます</param>
        /// <param name="containerElementPaths">指定された要素について、元になる XML の要素に対して他の XML ファイルの要素を子要素として追加します</param>
        /// <param name="otherSourceFiles">追加する内容が含まれた他の XML ファイル</param>
        private static void AddOtherXmlContentImpl(
            string destinationFile,
            string baseSourceFile,
            IEnumerable<string> valueElementPaths,      // XPath のサブセット
            IEnumerable<string> containerElementPaths,  // XPath のサブセット
            IEnumerable<string> otherSourceFiles)
        {
            var valueElementOperations = valueElementPaths.Select(path => new AddOperation(AddOperationType.ValueElement, path));
            var containerElementoperations = containerElementPaths.Select(path => new AddOperation(AddOperationType.ContainerElement, path));
            var operations = valueElementOperations.Concat(containerElementoperations).ToArray();

            var xml = XDocument.Load(baseSourceFile);

            foreach (var sourceFile in otherSourceFiles)
            {
                AddOtherXmlContentImpl(xml, operations, sourceFile);
            }

            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Indent = true;
            settings.IndentChars = "    ";

            using (XmlWriter xmlWriter = XmlWriter.Create(destinationFile, settings))
            {
                xml.Save(xmlWriter);
            }
        }
    }
}
