﻿using Microsoft.Build.Construction;
using Microsoft.Build.Evaluation;
using Nintendo.Nact;
using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.IO;
using System.Linq;
using System.Text;
using System.Web.Script.Serialization;
using System.Xml.Linq;
using YamlDotNet.RepresentationModel;

namespace SigloNact.BuiltIns.Bundle
{
    [NactConstants]
    public class ExecutableProductInfo : INactObject
    {
        public ImmutableArray<object> NactObjectCreationArguments { get; }
        public string Name { get; }
        public string Platform { get; }
        public string BuildType { get; }
        public IEnumerable<FilePath> FilesForBundle { get; }

        [NactObjectCreator]
        public ExecutableProductInfo(
            string name,
            string platform,
            string buildType,
            IEnumerable<FilePath> filesForBundle)
        {
            this.NactObjectCreationArguments = ImmutableArray.Create<object>(
                name, platform, buildType, filesForBundle);
            this.Name = name;
            this.Platform = platform;
            this.BuildType = buildType;
            this.FilesForBundle = filesForBundle;
        }
    }

    public class TestFramework
    {
        public string Name { get; }
        public string RunnerInterfaceVersion { get; }

        public TestFramework(string name, string runnerInterfaceVersion)
        {
            this.Name = name;
            this.RunnerInterfaceVersion = runnerInterfaceVersion;
        }
    }

    [NactAction]
    public class CreatePcToolExecutableProductInfoFile : INactAction
    {
        private FilePath m_SolutionFile { get; }
        private FilePath m_ProjectFile { get; }
        private string m_ToolsVersion { get; }
        private IEnumerable<FilePath> m_Resources { get; }
        private FilePath m_Root { get; }
        private string m_ProjectPlatform { get; }
        private string m_ProjectConfiguration { get; }
        private FilePath m_Output { get; }

        [NactFunction]
        public IEnumerable<object> SkipIfNotModified => NactBuiltInUtil.MakeDefaultSkipIfNotModified(this);
        [NactFunction]
        public string ProjectPlatform => m_ProjectPlatform;
        [NactFunction]
        public string ProjectConfiguration => m_ProjectConfiguration;
        [NactFunction]
        public FilePath Output => m_Output;
        public ImmutableArray<object> NactObjectCreationArguments { get; }

        [NactObjectCreator]
        public CreatePcToolExecutableProductInfoFile(
            FilePath outputDirectory,
            FilePath solutionFile,
            FilePath projectFile,
            string platform,
            string configuration,
            string toolsVersion,
            IEnumerable<FilePath> resources,
            FilePath root)
        {
            NactObjectCreationArguments = ImmutableArray.Create<object>(
                outputDirectory,
                solutionFile,
                projectFile,
                platform,
                configuration,
                toolsVersion,
                resources,
                root);
            this.m_SolutionFile = solutionFile;
            this.m_ProjectFile = projectFile;
            this.m_ToolsVersion = toolsVersion;
            this.m_Resources = resources;
            this.m_Root = root;

            {
                var solution = SolutionFile.Parse(solutionFile.PathString);
                ProjectInSolution projectInSolution;
                try
                {
                    projectInSolution = solution.ProjectsInOrder
                        .Single(p => FilePath.CreateLocalFileSystemPath(p.AbsolutePath) == projectFile);
                }
                catch (InvalidOperationException e)
                {
                    throw new ErrorException($"project '{projectFile.PathString}' is not found in {solutionFile.PathString}. ", string.Empty, e);
                }
                var projectConfig = projectInSolution.ProjectConfigurations[$"{configuration}|{platform}"];

                this.m_ProjectPlatform = projectConfig.PlatformName;
                this.m_ProjectConfiguration = projectConfig.ConfigurationName;
                this.m_Output = outputDirectory.Combine($"{projectFile.FileNameWithoutExtension}.{projectConfig.PlatformName}.{projectConfig.ConfigurationName}.yml");
            }
        }

        [NactActionFunction]
        public NactActionResult Execute(INactActionContext context)
        {
            var helper = context.Helper;
            var encoding = new UTF8Encoding(true);

            // ToDo: Nact.Core の MSBuildUtil を整理して使用する
            var globalProperties = new Dictionary<string, string>()
            {
                ["BuildingSolutionFile"] = "true",
                ["SolutionDir"] = m_SolutionFile.PathString + Path.DirectorySeparatorChar,
                ["SolutionExt"] = m_SolutionFile.Extension,   // .sln
                ["SolutionFileName"] = m_SolutionFile.Leaf.OriginalValue,
                ["SolutionName"] = Path.GetFileNameWithoutExtension(m_SolutionFile.Leaf.OriginalValue),
                ["SolutionPath"] = m_SolutionFile.PathString,
                ["Configuration"] = m_ProjectConfiguration,
                ["Platform"] = m_ProjectPlatform,
            };
            var project = ProjectCollection.GlobalProjectCollection.LoadProject(m_ProjectFile.PathString, globalProperties, m_ToolsVersion);

            // プロジェクトをロードして、テストに対応する実行ファイルのパスと TestFramework を判定
            var targetPath = FilePath.CreateLocalFileSystemPath(project.GetPropertyValue("TargetPath"));

            var executableFileMap = new Dictionary<string, string>()
            {
                { "Format", GetFormat(targetPath) },
                { "Path", m_Root.GetRelativeFilePathTo(targetPath).PathString },
            };
            var map = new Dictionary<string, object>()
            {
                { "Name", m_ProjectFile.FileNameWithoutExtension },
                { "Project", m_Root.GetRelativeFilePathTo(m_ProjectFile).PathString },
                { "Architecture", m_ProjectPlatform }, // TORIAEZU
                { "Os", "win32" },
                { "Hardware", "pc" },
                { "BuildType", m_ProjectConfiguration },
                { "ExecutableFiles", new[] { executableFileMap } },
                { "TestFrameworks", GetTestFrameworks(project, helper) },
                { "Resources", m_Resources.Select(path => m_Root.GetRelativeFilePathTo(path).PathString) }
            };

            var serializer = new YamlDotNet.Serialization.Serializer();
            using (var sw = new StreamWriter(m_Output.PathString, false, encoding))
            {
                serializer.Serialize(sw, map);
            }
            helper.AddWriteFile(m_Output);

            return helper.FinishAsSuccess();
        }

        private string GetFormat(FilePath path)
        {
            switch (path.Extension)
            {
                case ".exe":
                    return "Exe";
                case ".dll":
                    return "Dll";
                default:
                    throw new InternalLogicErrorException($"should never be reached.");
            }
        }

        private static IEnumerable<TestFramework> GetTestFrameworks(Project project, INactActionHelper helper)
        {
            // NuGet パッケージの参照とバージョンの組み合わせを取得
            var referenceVersionMap = new Dictionary<string, string>();
            foreach (var reference in GetReferenceNameAndVersions(project, helper))
            {
                var formattedVersion = FormatVersion(reference.Version);
                if (!referenceVersionMap.TryGetValue(reference.Name, out var prevVersion))
                {
                    referenceVersionMap.Add(reference.Name, formattedVersion);
                }
                else if (prevVersion != formattedVersion)
                {
                    throw new ErrorException($"Multiple version references to '{reference.Name}' exist. (Version1: '{prevVersion}', Version2: '{formattedVersion}')");
                }
            }

            // Reference Item からも参照とバージョンを取得
            foreach (var reference in GetReferenceNameAndVersionsForReferenceItem(project, helper))
            {
                var formattedVersion = FormatVersion(reference.Version);
                // NuGet パッケージから同名の参照を既に取得している場合は無視
                if (!referenceVersionMap.TryGetValue(reference.Name, out var prevVersion))
                {
                    referenceVersionMap.Add(reference.Name, formattedVersion);
                }
            }

            // MSTest
            if (referenceVersionMap.ContainsKey("Microsoft.VisualStudio.QualityTools.UnitTestFramework")
                || referenceVersionMap.ContainsKey("Microsoft.VisualStudio.TestPlatform.TestFramework"))
            {
                yield return new TestFramework("MSTest", "1.0.0.0");
            }

            // xUnit
            if (referenceVersionMap.ContainsKey("xunit.abstractions"))
            {
                yield return new TestFramework("xUnit", referenceVersionMap["xunit.abstractions"]);
            }

            // GoogleTest
            ProjectItemDefinition linkItem;
            if (project.ItemDefinitions.TryGetValue("Link", out linkItem))
            {
                var linkDependencies = linkItem.GetMetadataValue("AdditionalDependencies").Split(';');
                if (linkDependencies.Contains("libgtest.lib"))
                {
                    yield return new TestFramework("GoogleTest", "1.8.0.0");
                }
            }
        }

        private static IEnumerable<(string Name, string Version)> GetReferenceNameAndVersions(Project project, INactActionHelper helper)
        {
            // packages.config
            if (project.Items.Any(x => x.EvaluatedInclude == "packages.config"))
            {
                var packagesConfigPath = FilePath.CreateLocalFileSystemPath(Path.Combine(project.DirectoryPath, "packages.config"));
                XElement[] packages;
                using (var fs = helper.OpenRead(packagesConfigPath))
                {
                    var xml = XDocument.Load(fs);
                    packages = xml.Element("packages").Elements("package").ToArray();
                }

                foreach (var p in packages)
                {
                    yield return (p.Attribute("id").Value, p.Attribute("version")?.Value);
                }
            }

            // PackageReference
            var packageReferenceItems = project.Items.Where(x => x.ItemType == "PackageReference");
            foreach (var packageReferenceItem in packageReferenceItems)
            {
                yield return (packageReferenceItem.EvaluatedInclude, packageReferenceItem.GetMetadataValue("Version"));
            }

            // project.assets.json
            if (packageReferenceItems.Any())
            {
                var baseIntermediateOutputPath = project.GetPropertyValue("BaseIntermediateOutputPath");
                if (string.IsNullOrEmpty(baseIntermediateOutputPath))
                {
                    throw new ErrorException($"Property 'BaseIntermediateOutputPath' is not found in project '{project.FullPath}'.");
                }
                // デフォルトでは BaseIntermediateOutputPath の値はプロジェクトのディレクトリからの相対パス
                if (!Path.IsPathRooted(baseIntermediateOutputPath))
                {
                    baseIntermediateOutputPath = Path.Combine(project.DirectoryPath, baseIntermediateOutputPath);
                }
                var projectAssetsPath = FilePath.CreateLocalFileSystemPath(Path.Combine(baseIntermediateOutputPath, "project.assets.json"));

                Dictionary<string, object> projectAssetsData;
                var serializer = new JavaScriptSerializer();
                try
                {
                    projectAssetsData = serializer.Deserialize<Dictionary<string, object>>(helper.ReadAllText(projectAssetsPath, new UTF8Encoding(true)));
                }
                catch (Exception e) when (e is FileNotFoundException || e is DirectoryNotFoundException)
                {
                    // パスが誤っているか、実行情報ファイル生成前に NuGet 復元が行われていない
                    throw new ErrorException($"Failed to read '{projectAssetsPath}'.", string.Empty, e);
                }
                if (projectAssetsData.TryGetValue("libraries", out var libraries))
                {
                    foreach (var lib in ((Dictionary<string, object>)libraries).Keys)
                    {
                        var items = lib.Split('/');
                        if (items.Length != 2)
                        {
                            throw new ErrorException($"'{lib}' is unexpected format. (File: '{projectAssetsPath}')");
                        }
                        yield return (items[0], items[1]);
                    }
                }
            }
        }

        private static IEnumerable<(string Name, string Version)> GetReferenceNameAndVersionsForReferenceItem(Project project, INactActionHelper helper)
        {
            // Reference
            foreach (var referenceItem in project.Items.Where(x => x.ItemType == "Reference"))
            {
                var includes = referenceItem.EvaluatedInclude.Split(',').Select(x => x.Trim()).ToArray();
                yield return (includes[0], includes.FirstOrDefault(x => x.StartsWith("Version="))?.Replace("Version=", string.Empty));
            }
        }

        // バージョン値を '<major>.<minor>.<micro>.<revision>' に揃え、不足している要素は 0 で埋める
        private static string FormatVersion(string version)
        {
            if (string.IsNullOrEmpty(version))
            {
                // TORIAEZU: Reference からはバージョンを取得できない場合がある
                return string.Empty;
            }

            const int versionComponentLength = 4;
            // 各要素に数値以外の値が連結されていても無視する （'-pre' など）
            var components = version.Split('.')
                .Select(x => new string(x.TakeWhile(c => char.IsNumber(c)).ToArray())).ToArray();

            if (components.Length < versionComponentLength)
            {
                components = components.Concat(Enumerable.Repeat("0", versionComponentLength - components.Length)).ToArray();
            }
            return string.Join(".", components, 0, versionComponentLength);
        }
    }

    [NactActionFunctionContainer]
    public static class CreateTargetExecutableProductInfoFileContainer
    {
        [NactActionFunction]
        public static NactActionResult CreateTargetExecutableProductInfoFile(
            INactActionContext context,
            FilePath output,
            string name,
            string architecture,
            string os,
            string hardware,
            string buildType,
            IEnumerable<IReadOnlyDictionary<string, string>> executableFiles,
            FilePath projectFile,
            IEnumerable<FilePath> resources,
            IEnumerable<string> linkLibraryNames,
            FilePath root)
        {
            var helper = context.Helper;
            var encoding = new UTF8Encoding(true);
            var map = new Dictionary<string, object>()
            {
                { "Name", name },
                { "Project", projectFile != null ? root.GetRelativeFilePathTo(projectFile).PathString : null },
                { "Architecture", architecture },
                { "Os", os },
                { "Hardware", hardware },
                { "BuildType", buildType },
                { "ExecutableFiles", executableFiles.Select(x => x.ToDictionary(y => y.Key, y => y.Value)) },
                { "TestFrameworks", GetTestFrameworks(linkLibraryNames) },
                { "Resources", resources.Select(path => root.GetRelativeFilePathTo(path).PathString) }
            };

            var serializer = new YamlDotNet.Serialization.Serializer();
            using (var sw = new StreamWriter(output.PathString, false, encoding))
            {
                serializer.Serialize(sw, map);
            }
            helper.AddWriteFile(output);

            return helper.FinishAsSuccess();
        }

        private static IEnumerable<TestFramework> GetTestFrameworks(IEnumerable<string> linkLibraryNames)
        {
            if (linkLibraryNames.Contains("libgtest") || linkLibraryNames.Contains("libnnt_gtest"))
            {
                yield return new TestFramework("GoogleTest", "1.8.0.0");
            }
        }
    }
}
