﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Utilities;
using SigloNact.Utilities;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using static System.FormattableString;

namespace SigloNact.BuiltIns.BuildSystem
{
    [NactActionFunctionContainer]
    public static class GitActions
    {
        [NactActionFunction]
        public static NactActionResult MakeGitFileList(
            INactActionContext context,
            FilePath destination,
            FilePath repositoryRoot,
            FilePath revisionInfoFile)
        {
            var helper = context.Helper;

            // リビジョンが変更されたら再生成するようにするため、revisionInfoFile に対してダミーの読み取りを行う。
            // 本来はリビジョンを指定して git 管理ファイルを列挙すべきであるので、そうする。
            var revisionInfo = DataContractSerializationUtil.DeserializeFromBytes<GitRevisionInfo>(helper.ReadAllBytes(revisionInfoFile));

            var gitFiles = new GitInfo(repositoryRoot).Files
                .Select(x => repositoryRoot.GetRelativeFilePathTo(x).PathString.Replace("\\", "/")); // パス区切り文字をスラッシュに統一
            // svn も含まれるため exrepo 管理のファイルのリストアップは GitExecuter で行う
            var exrepoFiles = new GitExecuter(context.ProgramExecuter, context.EnvironmentVariables).GitLsExrepoFiles(repositoryRoot.PathString);
            // 書き出し
            try
            {
                helper.WriteAllLines(destination, gitFiles.Concat(exrepoFiles).ToArray(), Encoding.UTF8);
                return helper.FinishAsSuccess();
            }
            catch (Exception e) when (
                e is Nintendo.Nact.ErrorException
                || ExceptionUtil.IsIORelatedException(e))
            {
                helper.SetErrorSummary(string.Format(Strings.FailedToWriteGitFileList, destination.PathString, e.Message));
                return helper.FinishAsFailure();
            }
        }

        [NactActionFunction(Version = 1)]
        public static NactActionResult MakeGitRevisionInfo(
            INactActionContext context,
            FilePath outputFile,
            FilePath repositoryRoot,
            string revision)
        {
            var helper = context.Helper;

            var info = GitUtilities.GetGitRevisionInfo(repositoryRoot.PathString, revision);
            var contents = DataContractSerializationUtil.SerializeToBytes(info);

            var outputFilePathString = outputFile.PathString;
            helper.AddWriteFile(outputFile);
            // ファイル内容に変化がなければファイルを更新しないようにする（後続を再ビルドしないようにする）
            if (!Util.FileHasContents(outputFilePathString, contents))
            {
                File.WriteAllBytes(outputFilePathString, contents);
            }

            return helper.FinishAsSuccess();
        }

        private static IReadOnlyCollection<string> MakeBuildRevisionHeaderFileHeaderLines = new string[]
        {
            @"/*--------------------------------------------------------------------------------*",
            @"  Copyright (C)Nintendo All rights reserved.",
            @"",
            @"  These coded instructions, statements, and computer programs contain proprietary",
            @"  information of Nintendo and/or its licensed developers and are protected by",
            @"  national and international copyright laws. They may not be disclosed to third",
            @"  parties or copied or duplicated in any form, in whole or in part, without the",
            @"  prior written consent of Nintendo.",
            @"",
            @"  The content herein is highly confidential and should be handled accordingly.",
            @" *--------------------------------------------------------------------------------*/",
            @"",
            @"#pragma once",
            @"",
            @"// DO NOT EDIT; this file was generated by the build system.",
            @"",
            @"// Infomation on the revision where the build is being performed.",
            @"//",
            @"// IMPORTANT NOTE:",
            @"//     This header file will not be included in the SDK. You cannot include this file",
            @"//     from public header files and source files.",
            @"//",
            @"//     Including this header file causes the source file to be recompiled on every",
            @"//     revision change (~= every commit and checkout), which increases build time.",
            @"//     Use only if your module really needs revision information, for example:",
            @"//",
            @"//     - Your tool is not contained in the firmware but needs to show the revision",
            @"//       information to developers.",
            @"",
        };

        [NactActionFunction(Version = 1)]
        public static NactActionResult MakeBuildRevisionHeaderFile(
            INactActionContext context,
            FilePath outputFile,
            FilePath revisionInfoFile)
        {
            var helper = context.Helper;

            var revisionInfo = DataContractSerializationUtil.DeserializeFromBytes<GitRevisionInfo>(helper.ReadAllBytes(revisionInfoFile));
            var dateString = revisionInfo.CommitterDate.ToString("yyyy'-'MM'-'dd");
            var timeString = revisionInfo.CommitterDate.ToString("HH':'mm':'ss");

            using (var sw = new StreamWriter(helper.CreateFile(outputFile)))
            {
                foreach (var line in MakeBuildRevisionHeaderFileHeaderLines)
                {
                    sw.WriteLine(line);
                }

                sw.WriteLine("// Full revision hash string (currently 40 characters long).");
                sw.WriteLine("#define NN_BUILD_REVISION_HASH_STRING_FULL \"{0}\"", revisionInfo.CommitHash);
                sw.WriteLine();
                sw.WriteLine("// Short revision hash string (currently 10 characters long).");
                sw.WriteLine("#define NN_BUILD_REVISION_HASH_STRING_SHORT \"{0}\"", revisionInfo.CommitHash.Substring(0, 10));
                sw.WriteLine();
                sw.WriteLine("// Committer date and time string in ISO 8601 (UTC).");
                sw.WriteLine("#define NN_BUILD_REVISION_COMMITTER_DATETIME_STRING \"{0}T{1}Z\"", dateString, timeString);
                sw.WriteLine();
                sw.WriteLine("// Committer date string (UTC).");
                sw.WriteLine("#define NN_BUILD_REVISION_COMMITTER_DATE_STRING \"{0}\"", dateString);
                sw.WriteLine();
                sw.WriteLine("// Committer time string (UTC).");
                sw.WriteLine("#define NN_BUILD_REVISION_COMMITTER_TIME_STRING \"{0}\"", timeString);
            }

            return helper.FinishAsSuccess();
        }
    }
}
