﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using Microsoft.Build.Evaluation;

namespace Nintendo.MakeVisualStudioProject.Converter
{
    internal class Win32ItemMetadataConverter : ItemMetadataConverter
    {
        public Win32ItemMetadataConverter(IEnumerable<OptionDefinition> ods)
        {
            this.OptionDefinitions = ods;
        }

        // link.exe のオプション '/MANIFESTUAC:' は特殊対応が必要 (詳細は Win32OptionConverter のコメント参照)
        // なお、cl.exe, lib.exe に '/MANIFESTUAC:' なるオプションは存在しない
        protected override IEnumerable<OptionValue> ConvertToOptionValue(
            ItemMetadata im, IEnumerable<ItemMetadata> all, OptionDefinition od, out IEnumerable<ItemMetadata> consumed)
        {
            if (od.Switch == "/MANIFESTUAC:")
            {
                return ConvertToEnableUACOptionValue(im, all, out consumed);
            }
            else
            {
                return base.ConvertToOptionValue(im, all, od, out consumed);
            }
        }
        protected override IEnumerable<ItemMetadata> ConvertToItemMetadata(
            OptionValue ov, IEnumerable<OptionValue> all, OptionDefinition od, out IEnumerable<OptionValue> consumed)
        {
            if (od.Switch == "/MANIFESTUAC:")
            {
                return ConvertToEnableUACItemMetadata(ov, all, out consumed);
            }
            else
            {
                return base.ConvertToItemMetadata(ov, all, od, out consumed);
            }
        }

        private IEnumerable<OptionValue> ConvertToEnableUACOptionValue(
            ItemMetadata im, IEnumerable<ItemMetadata> all, out IEnumerable<ItemMetadata> consumed)
        {
            var consumedList = new List<ItemMetadata>();
            consumed = consumedList;

            if (im.Name != "EnableUAC")
            {
                return null;
            }

            consumedList.Add(im);

            var uacExecutionLevel = default(string);
            {
                var uacExecutionLevelItemMetadatas = all.Where(x => x.Name == "UACExecutionLevel");
                if (uacExecutionLevelItemMetadatas.Any())
                {
                    // TORIAEZU: もし複数のメタデータがあれば、最後のメタデータの値を採用
                    foreach (var metadata in uacExecutionLevelItemMetadatas)
                    {
                        switch (metadata.Value)
                        {
                            case "AsInvoker":
                                uacExecutionLevel = "level='asInvoker'";
                                break;
                            case "HighestAvailable":
                                uacExecutionLevel = "level='highestAvailable'";
                                break;
                            case "RequireAdministrator":
                                uacExecutionLevel = "level='requireAdministrator'";
                                break;
                            default:
                                throw new NotImplementedException();
                        }
                    }
                    consumedList.AddRange(uacExecutionLevelItemMetadatas);
                }
            }
            var uacUIAccess = default(string);
            {
                var uacUIAccessItemMetadatas = all.Where(x => x.Name == "UACUIAccess");
                if (uacUIAccessItemMetadatas.Any())
                {
                    // TORIAEZU: もし複数のメタデータがあれば、最後のメタデータの値を採用
                    foreach (var metadata in uacUIAccessItemMetadatas)
                    {
                        switch (metadata.Value)
                        {
                            case "true":
                                uacUIAccess = "uiAccess='true'";
                                break;
                            case "false":
                                uacUIAccess = "uiAccess='false'";
                                break;
                            default:
                                throw new NotImplementedException();
                        }
                    }
                    consumedList.AddRange(uacUIAccessItemMetadatas);
                }
            }

            // ふたつの引数は共に必須引数
            if (uacExecutionLevel == null || uacUIAccess == null)
            {
                return null;
            }

            return new OptionValue[] { new OptionValue("/MANIFESTUAC:", string.Join(" ", new[] { uacExecutionLevel, uacUIAccess })) };
        }

        private IEnumerable<ItemMetadata> ConvertToEnableUACItemMetadata(
            OptionValue ov, IEnumerable<OptionValue> all, out IEnumerable<OptionValue> consumed)
        {
            consumed = new OptionValue[] { ov };

            if (ov.Switch != "/MANIFESTUAC:")
            {
                return null;
            }
            if (!ov.HasArgument)
            {
                return null;
            }

            var uacExecutionLevel = default(ItemMetadata);
            var uacUIAccess = default(ItemMetadata);

            var arguments = ov.Argument.Split(new[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);

            foreach (var arg in arguments)
            {
                switch (arg)
                {
                    case "level='asInvoker'":
                        uacExecutionLevel = new ItemMetadata("UACExecutionLevel", "AsInvoker");
                        break;
                    case "level='highestAvailable'":
                        uacExecutionLevel = new ItemMetadata("UACExecutionLevel", "HighestAvailable");
                        break;
                    case "level='requireAdministrator'":
                        uacExecutionLevel = new ItemMetadata("UACExecutionLevel", "RequireAdministrator");
                        break;
                    case "uiAccess='true'":
                        uacUIAccess = new ItemMetadata("UACUIAccess", "true");
                        break;
                    case "uiAccess='false'":
                        uacUIAccess = new ItemMetadata("UACUIAccess", "false");
                        break;
                    default:
                        // 無視
                        break;
                }
            }

            // ふたつの引数は共に必須引数
            if (uacExecutionLevel == null || uacUIAccess == null)
            {
                return null;
            }
            else
            {
                return new ItemMetadata[] {
                    new ItemMetadata("EnableUAC", "true"), uacExecutionLevel, uacUIAccess
                };
            }
        }
    }
}
