﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.MakeVisualStudioProject.Converter
{
    public abstract class ItemMetadataConverter : IItemMetadataConverter
    {
        protected IEnumerable<OptionDefinition> OptionDefinitions { get; set; }

        // TODO: 同じような実装のメソッドを共通化する

        public IEnumerable<OptionValue> ConvertToOptionValue(IEnumerable<ItemMetadata> ims, out IEnumerable<ItemMetadata> unknowns)
        {
            var target = new List<ItemMetadata>(ims);
            var unknown = new List<ItemMetadata>();
            var converted = new List<OptionValue>();

            while (target.Count > 0)
            {
                var im = target.First();

                IEnumerable<ItemMetadata> consumed;
                var optionValues = ConvertToOptionValue(im, target, out consumed);

                if (optionValues != null)
                {
                    converted.AddRange(optionValues);
                    foreach (var c in consumed.ToArray())
                    {
                        target.Remove(c);
                    }
                }
                else
                {
                    target.Remove(im);
                    unknown.Add(im);
                }
            }

            unknowns = unknown;
            return converted;
        }

        public IEnumerable<ItemMetadata> ConvertToItemMetadata(IEnumerable<OptionValue> ovs, out IEnumerable<OptionValue> unknowns)
        {
            var target = new List<OptionValue>(ovs);
            var unknown = new List<OptionValue>();
            var converted = new List<ItemMetadata>();

            while (target.Count > 0)
            {
                var ov = target.First();

                IEnumerable<OptionValue> consumed;
                var itemMetadatas = ConvertToItemMetadata(ov, target, out consumed);

                if (itemMetadatas != null)
                {
                    converted.AddRange(itemMetadatas);
                    foreach (var c in consumed.ToArray())
                    {
                        target.Remove(c);
                    }
                }
                else
                {
                    target.Remove(ov);
                    unknown.Add(ov);
                }
            }

            unknowns = unknown;
            return converted;
        }

        protected virtual IEnumerable<ItemMetadata> ConvertToItemMetadata(
            OptionValue ov, IEnumerable<OptionValue> all, out IEnumerable<OptionValue> consumed)
        {
            foreach (var od in OptionDefinitions)
            {
                var itemMetadatas = ConvertToItemMetadata(ov, all, od, out consumed);
                if (itemMetadatas != null)
                {
                    return itemMetadatas;
                }
            }

            consumed = new OptionValue[] { ov };
            return null;
        }

        protected virtual IEnumerable<ItemMetadata> ConvertToItemMetadata(
            OptionValue ov, IEnumerable<OptionValue> all, OptionDefinition od, out IEnumerable<OptionValue> consumed)
        {
            consumed = new OptionValue[] { ov };

            if (ov.Switch != od.Switch)
            {
                return null;
            }

            if (!od.AcceptsArgument)
            {
                if (!ov.HasArgument)
                {
                    return new ItemMetadata[] { new ItemMetadata(od.MetadataName, od.MetadataValue) };
                }
                else
                {
                    return null;
                }
            }

            string argumentMetadataValue;
            if (od.AcceptsMultipleArguments)
            {
                IEnumerable<OptionValue> sameOptions;
                if (od.IsArgumentRequired)
                {
                    sameOptions = all.Where(x => (x.Switch == ov.Switch) && (x.HasArgument));
                    if (!sameOptions.Any())
                    {
                        return null;
                    }
                }
                else
                {
                    sameOptions = all.Where(x => x.Switch == ov.Switch);
                }

                consumed = sameOptions;

                var arguments = sameOptions.Where(x => x.HasArgument).Select(x => x.Argument);
                if (arguments.Any())
                {
                    // TODO: 末尾に %(MetadataName) を呼び出し元に通知せず入れてしまうのはよくないはずなので、要再検討
                    argumentMetadataValue = string.Format("{0}{1}%({2})",
                        string.Join(od.ArgumentSeparator, arguments),
                        od.ArgumentSeparator,
                        od.HasArgumentName ? od.ArgumentName : od.MetadataName);
                }
                else
                {
                    argumentMetadataValue = string.Empty;
                }
            }
            else
            {
                if (od.IsArgumentRequired && !ov.HasArgument)
                {
                    return null;
                }

                argumentMetadataValue = ov.Argument ?? string.Empty;
            }

            if (od.HasArgumentName)
            {
                if (string.IsNullOrEmpty(argumentMetadataValue))
                {
                    return new ItemMetadata[] {
                        new ItemMetadata(od.MetadataName, od.MetadataValue)
                    };
                }
                else
                {
                    return new ItemMetadata[] {
                        new ItemMetadata(od.MetadataName, od.MetadataValue),
                        new ItemMetadata(od.ArgumentName, argumentMetadataValue)
                    };
                }
            }
            else
            {
                return new ItemMetadata[] {
                    new ItemMetadata(od.MetadataName, argumentMetadataValue)
                };
            }
        }

        protected virtual IEnumerable<OptionValue> ConvertToOptionValue(
            ItemMetadata im, IEnumerable<ItemMetadata> all, out IEnumerable<ItemMetadata> consumed)
        {
            foreach (var od in OptionDefinitions)
            {
                var optionValues = ConvertToOptionValue(im, all, od, out consumed);
                if (optionValues != null)
                {
                    return optionValues;
                }
            }

            consumed = new ItemMetadata[] { im };
            return null;
        }

        protected virtual IEnumerable<OptionValue> ConvertToOptionValue(
            ItemMetadata im, IEnumerable<ItemMetadata> all, OptionDefinition od, out IEnumerable<ItemMetadata> consumed)
        {
            consumed = new ItemMetadata[] { im };

            if (im.Name != od.MetadataName)
            {
                return null;
            }

            if (!od.AcceptsArgument)
            {
                if (im.Value == od.MetadataValue)
                {
                    return new OptionValue[] { new OptionValue(od.Switch) };
                }
                else
                {
                    return null;
                }
            }

            var argumentString = default(string);
            if (od.HasArgumentName)
            {
                var argumentMetadata = all.FirstOrDefault(x => x.Name == od.ArgumentName);
                if (argumentMetadata != null)
                {
                    consumed = new ItemMetadata[] { im, argumentMetadata };
                    argumentString = argumentMetadata.Value;
                }
            }
            else
            {
                argumentString = im.Value;
            }

            if (od.AcceptsMultipleArguments)
            {
                var arguments = argumentString.Split(new[] { od.ArgumentSeparator }, StringSplitOptions.None);
                if (od.IsArgumentRequired && arguments.Any(arg => string.IsNullOrEmpty(arg)))
                {
                    return null;
                }

                return arguments
                    .Where(arg => arg != string.Format("%({0})", od.HasArgumentName ? od.ArgumentName : od.MetadataName))
                    .Select(arg => {
                        if (string.IsNullOrEmpty(arg))
                        {
                            return new OptionValue(od.Switch);
                        }
                        else
                        {
                            return new OptionValue(od.Switch, arg);
                        }
                    });
            }
            else
            {
                if (od.IsArgumentRequired && string.IsNullOrEmpty(argumentString))
                {
                    return null;
                }

                if (string.IsNullOrEmpty(argumentString))
                {
                    return new OptionValue[] { new OptionValue(od.Switch) };
                }
                else
                {
                    return new OptionValue[] { new OptionValue(od.Switch, argumentString) };
                }
            }
        }
    }
}
