﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using Microsoft.Build.Evaluation;
using Microsoft.VisualStudio.Setup.Configuration;
using YamlDotNet.Serialization;

namespace Nintendo.MakeVisualStudioProject.Converter
{
    public class OptionDefinitionGenerator
    {
        private const string VisualStudio15ChannelId = "VisualStudio.15.Release";
        private const int REGDB_E_CLASSNOTREG = unchecked((int)0x80040154);

        private class Argument
        {
            public string Name { get; private set; }
            public bool IsRequired { get; private set; }

            public static Argument CreateIfExists(XmlElement e)
            {
                var argument = e.GetElementsByTagName("Argument").Cast<XmlElement>().SingleOrDefault();
                return (argument != null) ? new Argument(argument) : null;
            }

            public Argument(XmlElement e)
            {
                Name = e.GetAttributeOrThrow("Property");
                IsRequired = bool.Parse(e.GetAttributeOrThrow("IsRequired"));
            }
        }

        public static IEnumerable<OptionDefinition> Generate(string xmlPath, IEnumerable<string> ignoreMetadataNames = null)
        {
            var xml = new XmlDocument();
            xml.Load(xmlPath);

            var doc = xml.DocumentElement;
            var ret = new List<OptionDefinition>();

            var switchPrefix = doc.GetAttributeOrThrow("SwitchPrefix");

            foreach (var element in doc)
            {
                var e = element as XmlElement;
                if (e == null)
                {
                    continue;
                }

                if (e.HasAttribute("IncludeInCommandLine") && !bool.Parse(e.GetAttribute("IncludeInCommandLine")))
                {
                    continue;
                }
                if (e.HasAttributeAndStringValue("Name"))
                {
                    if (ignoreMetadataNames != null)
                    {
                        if (ignoreMetadataNames.Contains(e.GetAttribute("Name")))
                        {
                            continue;
                        }
                    }
                }
                ret.AddRange(GenerateOptionDefinition(e, switchPrefix));
            }

            return ret;
        }

        private static IEnumerable<OptionDefinition> GenerateOptionDefinition(XmlElement e, string switchPrefix)
        {
            switch (e.Name)
            {
                case "BoolProperty":
                    return GenerateBoolOptionDefinition(e, switchPrefix);
                case "EnumProperty":
                    return GenerateEnumOptionDefinition(e, switchPrefix);
                case "IntProperty":
                    return GenerateIntOptionDefinition(e, switchPrefix);
                case "StringProperty":
                    return GenerateStringOptionDefinition(e, switchPrefix);
                case "StringListProperty":
                    return GenerateStringListOptionDefinition(e, switchPrefix);
                default:
                    return Enumerable.Empty<OptionDefinition>();
            }
        }

        private static OptionDefinition CreateOptionDefinition(string s, string mn, string mv, Argument a)
        {
            if (a != null)
            {
                return new OptionDefinition(s, mn, mv, true, requiresArg: a.IsRequired, argName: a.Name);
            }
            else
            {
                return new OptionDefinition(s, mn, mv);
            }
        }

        private static IEnumerable<OptionDefinition> GenerateBoolOptionDefinition(XmlElement e, string switchPrefix)
        {
            var ret = new List<OptionDefinition>();

            Argument arg = Argument.CreateIfExists(e);

            // TORIAEZU: Switch も ReverseSwitch も無い BoolProperty は無視する（BuildingInIde など）
            if (e.HasAttributeAndStringValue("Switch"))
            {
                ret.Add(CreateOptionDefinition(switchPrefix + e.GetAttributeOrThrow("Switch"), e.GetAttributeOrThrow("Name"), "true", arg));
            }
            if (e.HasAttributeAndStringValue("ReverseSwitch"))
            {
                ret.Add(CreateOptionDefinition(switchPrefix + e.GetAttributeOrThrow("ReverseSwitch"), e.GetAttributeOrThrow("Name"), "false", arg));
            }

            return ret;
        }

        private static IEnumerable<OptionDefinition> GenerateEnumOptionDefinition(XmlElement e, string switchPrefix)
        {
            return e.GetElementsByTagName("EnumValue").Cast<XmlElement>().Select(ev =>
            {
                Argument arg = Argument.CreateIfExists(ev);

                return CreateOptionDefinition(ev.GetSwitchAttributeWithPrefix(switchPrefix), e.GetAttributeOrThrow("Name"), ev.GetAttributeOrThrow("Name"), arg);
            });
        }

        private static IEnumerable<OptionDefinition> GenerateIntOptionDefinition(XmlElement e, string switchPrefix)
        {
            return new OptionDefinition[]
            {
                new OptionDefinition(
                    e.GetSwitchAttributeWithPrefix(switchPrefix),
                    e.GetAttributeOrThrow("Name"),
                    null,
                    true,
                    requiresArg: true,
                    separator: e.GetAttributeOrNull("Separator"))
            };
        }

        private static IEnumerable<OptionDefinition> GenerateStringOptionDefinition(XmlElement e, string switchPrefix)
        {
            // 引数チェックをしないなら、実は IntOption と同じ仕様
            return GenerateIntOptionDefinition(e, switchPrefix);
        }

        private static IEnumerable<OptionDefinition> GenerateStringListOptionDefinition(XmlElement e, string switchPrefix)
        {
            return new OptionDefinition[]
            {
                new OptionDefinition(
                    e.GetSwitchAttributeWithPrefix(switchPrefix),
                    e.GetAttributeOrThrow("Name"),
                    null,
                    true,
                    requiresArg: true,
                    separator: e.GetAttributeOrNull("Separator"),
                    argSeparator: ";")
            };
        }

        public static IEnumerable<OptionDefinition> LoadDefinitions(StreamReader reader)
        {
            var deserializer = new DeserializerBuilder()
                .Build();

            return deserializer.Deserialize<IEnumerable<OptionDefinition>>(reader);
        }

        public static void SaveDefinitions(StreamWriter writer, IEnumerable<OptionDefinition> optionDefinitions)
        {
            var serializer = new SerializerBuilder()
                .EmitDefaults()
                .Build();

            serializer.Serialize(writer, optionDefinitions);
        }

        public static string GetVCTargetsPath(string toolsVersion, string vsVersion)
        {
            switch (toolsVersion)
            {
                case "4.0":
                case "12.0":
                case "14.0":
                    return GetVCTargetsPathFromRegistry(toolsVersion, vsVersion);
                case "15.0":
                    return GetVCTargetsPathFromSetupConfigurationAPI(VisualStudio15ChannelId);
                default:
                    throw new OptionGenerationException($"無効な ToolsVersion です: {toolsVersion}");
            }
        }

        private static string GetVCTargetsPathFromSetupConfigurationAPI(string channelId)
        {
            try
            {
                var configuration = new SetupConfiguration();
                var e = configuration.EnumInstances();

                int fetched;
                var instances = new ISetupInstance2[1];
                do
                {
                    e.Next(1, instances, out fetched);
                    if (fetched > 0)
                    {
                        var ins = instances[0];
                        if (!ins.GetState().HasFlag(InstanceState.Complete))
                        {
                            continue;
                        }
                        if (!ChannelManifestIdStartsWith(channelId, ins))
                        {
                            continue;
                        }
                        if (ins.GetPackages()
                                // ツールチェインオプション生成に必要な xml ファイルはこの vsix に入っているらしい (Clang/C2 用の xml も含む)
                                // xml 以外のファイルも必要になったら、確認するパッケージを再検討しないといけない
                                .Where(x => x.GetType() == "Vsix")
                                .Any(x => x.GetId() == "Microsoft.VisualStudio.VC.MSBuild.Base.Resources"))
                        {
                            return ins.ResolvePath(@"Common7\IDE\VC\VCTargets");
                        }
                    }
                }
                while (fetched > 0);
            }
            catch (System.Runtime.InteropServices.COMException ex) when (ex.HResult == REGDB_E_CLASSNOTREG)
            {
                // インストールを検出できなかったと判断して何もしない
            }

            throw new OptionGenerationException("MSBuild のインストール先ディレクトリの検出に失敗しました。");
        }

        private static bool ChannelManifestIdStartsWith(string channelId, ISetupInstance2 ins)
        {
            const string ChannelManifestIdPropertyName = "channelManifestId";

            var props = ins.GetProperties();
            var names = props.GetNames();
            if (names.Contains(ChannelManifestIdPropertyName))
            {
                return props.GetValue(ChannelManifestIdPropertyName).ToString().StartsWith(channelId);
            }
            else
            {
                return false;
            }
        }

        # region nact.exe から実装を拝借

        private static string GetVCTargetsPathFromRegistry(string toolsVersion, string vsVersion)
        {
            // VCTargetsPath はレジストリの HKLM\SOFTWARE\Microsoft\MSBuild\ToolsVersions で以下のように定義されている
            //
            //    $([MSBuild]::ValueOrDefault('$(VCTargetsPath)','$(MSBuildExtensionsPath32)\Microsoft.Cpp\v4.0\V***'))
            //
            // ただし、これをそのままプロパティとして評価してしまうと、環境によっては VCTargetsPath が既に定義されていて
            // 正しく評価が行えないことがあるため、上記の第二引数を決め打ちで抜き出して評価してしまうことにする

            var propertyValue = default(string);
            if (vsVersion == "100")
            {
                propertyValue = @"$(MSBuildExtensionsPath32)\Microsoft.Cpp\v4.0\";
            }
            else
            {
                propertyValue = string.Format(@"$(MSBuildExtensionsPath32)\Microsoft.Cpp\v4.0\V{0}\", vsVersion);
            }

            Project project;
            try
            {
                project = new Project(null, toolsVersion, ProjectCollection.GlobalProjectCollection);
            }
            catch (Microsoft.Build.Exceptions.InvalidProjectFileException ex)
            {
                throw new OptionGenerationException("MSBuild のインストール先ディレクトリの検出に失敗しました。", ex);
            }

            var propertyName = "MyVCTargetsPath";
            project.SetProperty(propertyName, propertyValue);
            project.ReevaluateIfNecessary();

            return project.GetPropertyValue(propertyName);
        }

        #endregion
    }
}
