﻿using Microsoft.Build.Construction;
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.MakeVisualStudioProject.Converter
{
    internal abstract class NXPlatformElementConverter : PlatformElementConverter, IBuiltinOptionDefinitionsAvailable
    {
        private Lazy<NXOptionConverter> m_CompilerOptionConverter;
        private Lazy<NXOptionConverter> m_LibrarianOptionConverter;
        private Lazy<NXOptionConverter> m_LinkerOptionConverter;

        protected abstract string Platform { get; }

        public bool UseBuiltinOptionDefinitions { get; set; }

        public NXPlatformElementConverter() : base()
        {
            m_CompilerOptionConverter = new Lazy<NXOptionConverter>(() =>
            {
                if (UseBuiltinOptionDefinitions)
                {
                    return NXOptionConverter.GetBuiltInCompilerOptionConverter(Platform);
                }
                else
                {
                    return NXOptionConverter.GetCompilerOptionConverter(Platform);
                }
            });
            m_LibrarianOptionConverter = new Lazy<NXOptionConverter>(() =>
            {
                if (UseBuiltinOptionDefinitions)
                {
                    return NXOptionConverter.GetBuiltInLibrarianOptionConverter(Platform);
                }
                else
                {
                    return NXOptionConverter.GetLibrarianOptionConverter(Platform);
                }
            });
            m_LinkerOptionConverter = new Lazy<NXOptionConverter>(() =>
            {
                if (UseBuiltinOptionDefinitions)
                {
                    return NXOptionConverter.GetBuiltInLinkerOptionConverter(Platform);
                }
                else
                {
                    return NXOptionConverter.GetLinkerOptionConverter(Platform);
                }
            });
        }

        public override bool AcceptsBuildTargetSetting(BuildTargetSetting targetSetting)
        {
            return targetSetting.Platform == Platform;
        }

        public override void AddCustomProperties(ProjectPropertyGroupElement propertyGroup, BuildTargetSetting targetSetting)
        {
            if (!string.IsNullOrEmpty(targetSetting.ApplicationDataDirectory))
            {
                propertyGroup.AddProperty("ApplicationDataDir", ConvertToSubstitutedPathOrRelativePath(targetSetting.ApplicationDataDirectory, targetSetting));
            }
            if (!string.IsNullOrEmpty(targetSetting.NroDeployDirectory))
            {
                propertyGroup.AddProperty("NRODeployPath", ConvertToSubstitutedPathOrRelativePath(targetSetting.NroDeployDirectory, targetSetting));
            }
        }

        public override void ReadCustomProperties(ProjectPropertyGroupElement propertyGroup, BuildTargetSetting targetSetting)
        {
            foreach (var property in propertyGroup.Properties)
            {
                switch (property.Name)
                {
                    case "ApplicationDataDir":
                        targetSetting.ApplicationDataDirectory = ConvertFromSubstitutedPathToAbsolutePath(property.Value, targetSetting);
                        break;
                    case "NRODeployPath":
                        targetSetting.NroDeployDirectory = ConvertFromSubstitutedPathToAbsolutePath(property.Value, targetSetting);
                        break;
                    default:
                        Log.WarnProjectElement(property,
                            "プロパティ '{0}' は未知のプロパティであるため、無視します。", property.Name);
                        break;
                }
            }
        }

        protected override void AddArchiveItemDefinition(ProjectItemDefinitionGroupElement itemDefinitionGroup, BuildTargetSetting targetSetting)
        {
            base.AddArchiveItemDefinition(itemDefinitionGroup, targetSetting);

            // 現状の Oasis は静的ライブラリの埋め込みに非対応のため、関連する ItemMetadata が存在する場合は警告を出して取り除く
            var lib = itemDefinitionGroup.ItemDefinitions.SingleOrDefault(x => x.ItemType == "Lib");
            if (lib != null)
            {
                foreach (var additionalDependencies in lib.Metadata.Where(x => x.Name == "AdditionalDependencies").ToArray())
                {
                    lib.RemoveChild(additionalDependencies);

                    Log.Warn($"プロジェクト構成 {targetSetting.Configuration}|{targetSetting.Platform} における LibraryFiles, LibraryNames, ObjectFiles の指定を無視します。"
                        + $"プラットフォーム {targetSetting.Platform} は %(Lib.AdditionalDependencies) を現在サポートしていません。");
                }
                foreach (var additionalLibraryDirectories in lib.Metadata.Where(x => x.Name == "AdditionalLibraryDirectories").ToArray())
                {
                    lib.RemoveChild(additionalLibraryDirectories);

                    Log.Warn($"プロジェクト構成 {targetSetting.Configuration}|{targetSetting.Platform} における LibrarySearchPath の指定を無視します。"
                        + $"プラットフォーム {targetSetting.Platform} は %(Lib.AdditionalLibraryDirectories) を現在サポートしていません。");
                }

                if (lib.Children.Count == 0)
                {
                    itemDefinitionGroup.RemoveChild(lib);
                }
            }
        }

        protected override void AddLinkItemDefinition(ProjectItemDefinitionGroupElement itemDefinitionGroup, BuildTargetSetting targetSetting)
        {
            base.AddLinkItemDefinition(itemDefinitionGroup, targetSetting);

            Lazy<ProjectItemDefinitionElement> link = new Lazy<ProjectItemDefinitionElement>(() =>
            {
                var itemDefinition = itemDefinitionGroup.ItemDefinitions.SingleOrDefault(x => x.ItemType == "Link");
                if (itemDefinition != null)
                {
                    return itemDefinition;
                }
                else
                {
                    return itemDefinitionGroup.AddItemDefinition("Link");
                }
            });

            if (targetSetting.SdkDynamicLinkLibraryNames.Any())
            {
                AddMultipleMetadataValues(link.Value, "AdditionalNSODependencies", targetSetting.SdkDynamicLinkLibraryNames);
            }
            if (targetSetting.ManualLoadDynamicLinkLibraryNames.Any())
            {
                AddMultipleMetadataValues(link.Value, "AdditionalNRODependencies", targetSetting.ManualLoadDynamicLinkLibraryNames);
            }
            if (!string.IsNullOrEmpty(targetSetting.LinkerType))
            {
                link.Value.AddMetadata("LinkerType", targetSetting.LinkerType);
            }
            if (!string.IsNullOrEmpty(targetSetting.DescFile))
            {
                link.Value.AddMetadata("FinalizeDescSource", ConvertToSubstitutedPathOrRelativePath(targetSetting.DescFile, targetSetting));
            }
            if (!string.IsNullOrEmpty(targetSetting.MetaFile))
            {
                link.Value.AddMetadata("FinalizeMetaSource", ConvertToSubstitutedPathOrRelativePath(targetSetting.MetaFile, targetSetting));
            }
            if (!string.IsNullOrEmpty(targetSetting.NrrFileName))
            {
                link.Value.AddMetadata("NRRFileName", ConvertToSubstitutedPathOrRelativePath(targetSetting.NrrFileName, targetSetting));
            }
        }

        protected override void ReadLinkItemDefinitions(ProjectItemDefinitionElement itemDefinition, BuildTargetSetting targetSetting)
        {
            base.ReadLinkItemDefinitions(itemDefinition, targetSetting);

            targetSetting.SdkDynamicLinkLibraryNames = ReadMultipleMetadataValues(itemDefinition, "AdditionalNSODependencies");
            targetSetting.ManualLoadDynamicLinkLibraryNames = ReadMultipleMetadataValues(itemDefinition, "AdditionalNRODependencies");

            var descSource = itemDefinition.Metadata.LastOrDefault(x => x.Name == "FinalizeDescSource");
            if (descSource != null)
            {
                targetSetting.DescFile = ConvertFromSubstitutedPathToAbsolutePath(descSource.Value, targetSetting);
            }

            var metaSource = itemDefinition.Metadata.LastOrDefault(x => x.Name == "FinalizeMetaSource");
            if (metaSource != null)
            {
                targetSetting.MetaFile = ConvertFromSubstitutedPathToAbsolutePath(metaSource.Value, targetSetting);
            }

            var nrrFileName = itemDefinition.Metadata.LastOrDefault(x => x.Name == "NRRFileName");
            if (nrrFileName != null)
            {
                targetSetting.NrrFileName = ConvertFromSubstitutedPathToAbsolutePath(nrrFileName.Value, targetSetting);
            }
        }

        protected override IEnumerable<ProjectMetadataElement> AddCompileOptionMetadatasFromOptionStrings(
            ProjectItemElement compileElement, IEnumerable<string> compileOptions, OptionConverter optionConverter)
        {
            var addedMetadatas = base.AddCompileOptionMetadatasFromOptionStrings(compileElement, compileOptions, optionConverter);
            return AddCompileOptionMetadatasFromOptionStringsImpl(compileElement, addedMetadatas);
        }

        protected override IEnumerable<ProjectMetadataElement> AddCompileOptionMetadatasFromOptionStrings(
            ProjectItemDefinitionElement compileElement, IEnumerable<string> compileOptions, OptionConverter optionConverter)
        {
            var addedMetadatas = base.AddCompileOptionMetadatasFromOptionStrings(compileElement, compileOptions, optionConverter);
            return AddCompileOptionMetadatasFromOptionStringsImpl(compileElement, addedMetadatas);
        }

        protected override OptionConverter GetCompileOptionConverter(BuildTargetSetting targetSetting)
        {
            return m_CompilerOptionConverter.Value;
        }

        protected override OptionConverter GetArchiveOptionConverter(BuildTargetSetting targetSetting)
        {
            return m_LibrarianOptionConverter.Value;
        }

        protected override OptionConverter GetLinkOptionConverter(BuildTargetSetting targetSetting)
        {
            return m_LinkerOptionConverter.Value;
        }

        protected override string CompilerName { get { return "clang.exe"; } }

        protected override string ArchiverName { get { return "ar.exe"; } }

        protected override string LinkerName { get { return "ld.exe"; } }

        protected override IEnumerable<string> ItemMetadataNamesExcludedFromArchiveOptionConversion
        {
            get
            {
                return base.ItemMetadataNamesExcludedFromArchiveOptionConversion
                    .Except(new string[] { "AdditionalDependencies", "AdditionalLibraryDirectories" });
            }
        }

        protected override IEnumerable<string> ItemMetadataNamesExcludedFromLinkOptionConversion
        {
            get
            {
                return base.ItemMetadataNamesExcludedFromLinkOptionConversion.Concat(s_AdditionalItemMetadataNamesExcludedFromLinkOptionConversion);
            }
        }
        private static readonly IEnumerable<string> s_AdditionalItemMetadataNamesExcludedFromLinkOptionConversion = new string[]
        {
            "FinalizeDescSource",
            "FinalizeMetaSource",
            "ApplicationDataDir",
            "AdditionalNSODependencies",
            "AdditionalNRODependencies",
            "NRODeployPath",
            "NRRFileName"
        };

        protected override string LibraryFileExtension { get { return ".a"; } }

        private IEnumerable<ProjectMetadataElement> AddCompileOptionMetadatasFromOptionStringsImpl(
            dynamic compileElement, IEnumerable<ProjectMetadataElement> addedMetadatas)
        {
            var addedMetadataList = new List<ProjectMetadataElement>(addedMetadatas);

            // 一部のメタデータは強制的に AdditionalOptions として保存する (現状 DisableSpecificWarnings のみ)
            var moreAdditionalOptions = new List<string>();

            var disableSpecificWarnings = addedMetadataList.Where(e => e.Name == "DisableSpecificWarnings").SingleOrDefault();
            if (disableSpecificWarnings != null)
            {
                addedMetadataList.Remove(disableSpecificWarnings);
                compileElement.RemoveChild(disableSpecificWarnings);

                foreach (var warning in disableSpecificWarnings.Value.Split(';'))
                {
                    if (warning != "%(DisableSpecificWarnings)")
                    {
                        moreAdditionalOptions.Add("-Wno-" + warning);
                    }
                }
            }

            if (moreAdditionalOptions.Any())
            {
                var optionString = string.Join(" ", moreAdditionalOptions);
                var additionalOptionsMetadata = addedMetadataList.SingleOrDefault(x => x.Name == "AdditionalOptions");

                if (additionalOptionsMetadata != null)
                {
                    additionalOptionsMetadata.Value += $" {optionString}";
                }
                else
                {
                    addedMetadataList.Add(
                        // %(AdditionalOptions) は *前に* 書く (-Wextra で有効化した警告を -Wno-* で無効化する、などの要求があるため)
                        compileElement.AddMetadata("AdditionalOptions", "%(AdditionalOptions) " + optionString));
                }
            }

            return addedMetadataList;
        }
    }
}
