﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using YamlDotNet.Core;
using YamlDotNet.Core.Events;
using YamlDotNet.Serialization;

namespace Nintendo.MakeSampleSolution
{
    /// <summary>
    /// ソリューション構成の選択方法の指定
    /// </summary>
    enum SolutionConfigurationPreference
    {
        /// <summary>
        /// ソリューションに含まれるプロジェクトすべてのプロジェクト構成の和集合
        /// </summary>
        All = 0,

        /// <summary>
        /// プロジェクト間の依存関係において根にあたるプロジェクトのプロジェクト構成の和集合
        /// </summary>
        OriginOnly,

        /// <summary>
        /// ソリューション設定ファイルで指定されたプロジェクトのプロジェクト構成の和集合
        /// </summary>
        //SelectedOnly,  // 未実装

        /// <summary>
        /// ソリューション設定ファイルで指定されるソリューション構成
        /// </summary>
        //Custom  // 未実装
    }

    /// <summary>
    /// ソリューション構成に対応するプロジェクト構成が見つからない時の解決方法の指定
    /// </summary>
    enum SolutionConfigurationResolutionPreference
    {
        /// <summary>
        /// 指定されたソリューション構成ではプロジェクトをビルドしない
        /// </summary>
        DoNotBuild = 0,

        /// <summary>
        /// 代替となるプロジェクト構成の自動選択を試行し、選択できなければエラー
        /// </summary>
        //AutoMap,  // 有用性が低いので公開しない

        /// <summary>
        /// ソリューション設定ファイルで指定されたプロジェクト構成の選択を試行し、選択できなければエラー
        /// </summary>
        CustomMap,

        /// <summary>
        /// エラーとする
        /// </summary>
        Error
    }

    internal class SolutionSetting
    {
        public string OutputName { get; set; }
        public IEnumerable<SolutionProject> Projects { get; set; }
        public Dictionary<string, string> SolutionProperties { get; set; }
        public SolutionConfigurationPreference SolutionConfigurationPreference { get; set; }
        public SolutionConfigurationResolutionPreference SolutionConfigurationResolutionPreference { get; set; }
        public IEnumerable<string> SolutionConfigurationResolutionRules { get; set; }

        public Dictionary<string, Guid> SolutionFolderPaths { get; private set; }

        public readonly static char[] s_SolutionFolderPathSeparators = new[] { '\\', '/' };

        private static Guid s_NamespaceGuid = new Guid("C447FE9F-B22C-4D0B-81BF-1B3691FFB5BC");

        public SolutionSetting()
            : this(Enumerable.Empty<SolutionProject>())
        {
        }
        public SolutionSetting(IEnumerable<SolutionProject> projects)
        {
            Projects = projects;
            SolutionFolderPaths = new Dictionary<string, Guid>();
        }

        public Guid GetSolutionFolderGuid(string solutionFolderPath)
        {
            if (string.IsNullOrWhiteSpace(solutionFolderPath))
            {
                return Guid.Empty;
            }
            else
            {
                var guid = Guid.Empty;
                SolutionFolderPaths.TryGetValue(solutionFolderPath, out guid);
                return guid;
            }
        }

        public static string GetSolutionFolderName(string path)
        {
            var index = path.LastIndexOfAny(s_SolutionFolderPathSeparators);
            if (index == -1)
            {
                return path;
            }
            else
            {
                return path.Substring(index + 1);
            }
        }

        public static string GetSolutionFolderDirectoryName(string path)
        {
            var index = path.LastIndexOfAny(s_SolutionFolderPathSeparators);
            if (index == -1)
            {
                return string.Empty;
            }
            else
            {
                return path.Substring(0, index);
            }
        }

        public static IEnumerable<SolutionSetting> ReadSolutionSettings(string path)
        {
            using (var stream = new StreamReader(path))
            {
                var deserializer = new Deserializer();
                var eventReader = new EventReader(new Parser(stream));

                eventReader.Expect<StreamStart>();

                while (eventReader.Accept<DocumentStart>())
                {
                    yield return deserializer.Deserialize<SolutionSetting>(eventReader).RebuildSolutionProjectsAndFolders();
                }
            }
        }

        private SolutionSetting RebuildSolutionProjectsAndFolders()
        {
            // ツリー構造を紐解いてプロジェクト一覧を列挙し、フォルダパスを親から子に追記する
            var projects = Flatten(Projects, x => x.Projects);
            // 親→子の順番で列挙されることを前提とした処理なので注意
            // Flatten は深さ優先探索で値を列挙するので問題ない
            foreach (var parent in projects)
            {
                if (!string.IsNullOrWhiteSpace(parent.SolutionFolderPath))
                {
                    foreach (var child in parent.Projects)
                    {
                        child.SolutionFolderPath = parent.SolutionFolderPath + @"\" + (child.SolutionFolderPath ?? string.Empty);
                    }
                }
            }

            // vcxproj のパスが設定された有効なプロジェクトだけを SolutionSetting に残す
            Projects = projects.Where(x => !string.IsNullOrWhiteSpace(x.Path));

            // ソリューションフォルダの一覧を構築
            SolutionFolderPaths = CreateSolutionFolderPaths();

            return this;
        }

        /// <summary>
        /// ソリューションフォルダのパスが設定されたプロジェクトからフォルダ一覧を構築します。
        /// </summary>
        /// <returns>ソリューションフォルダの一覧</returns>
        private Dictionary<string, Guid> CreateSolutionFolderPaths()
        {
            var solutionFolderPaths = new Dictionary<string, Guid>();
            foreach (var project in Projects.Where(x => !string.IsNullOrWhiteSpace(x.SolutionFolderPath)))
            {
                // ソリューションフォルダのパスを正規化する
                project.SolutionFolderPath = NormalizeSolutionFolderPath(project.SolutionFolderPath);

                var folderNames = new List<string>();
                // 正規化されたフォルダパスを分解して部分的なパスを構築、フォルダ一覧になければ追記する
                foreach (var name in project.SolutionFolderPath.Split(s_SolutionFolderPathSeparators))
                {
                    folderNames.Add(name);
                    var folderPath = string.Join(@"\", folderNames);
                    if (!solutionFolderPaths.ContainsKey(folderPath))
                    {
                        // TORIAEZU: フォルダパス＋vcxprojのパスから GUID を生成
                        solutionFolderPaths.Add(folderPath, CreateGuidFromStringWithSHA1(folderPath + project.Path));
                    }
                }
            }
            return solutionFolderPaths;
        }

        /// <summary>
        /// ソリューションフォルダーパスを正規化して返します。
        /// 不適切なフォルダ名が使用されている場合は例外 ArgumentException が発生します。
        /// </summary>
        /// <param name="solutionPath">ソリューションフォルダのパス</param>
        /// <returns>正規化されたソリューションフォルダのパス</returns>
        private static string NormalizeSolutionFolderPath(string solutionPath)
        {
            // 不適切なソリューションフォルダー名:
            // - 次の文字を含む名前: / ? : & \ * < > | # %
            // - Unicode コントロール文字を含む名前
            // - サロゲート文字を含む名前
            // - 'CON', 'AUX', 'PRN', 'COM1' または 'LPT2' のようなシステム予約名
            // - '.' または '..' という名前
            var invalidChars = Path.GetInvalidFileNameChars().Concat(new [] { '#', '%', '&' }).ToArray();
            return string.Join(@"\", solutionPath
                .Split(s_SolutionFolderPathSeparators)
                .Select(x => x.Trim())
                .Where(x => !string.IsNullOrWhiteSpace(x))
                .Select(x =>
                {
                    if (x.IndexOfAny(invalidChars) >= 0)
                    {
                        throw new ArgumentException(string.Format("ソリューションフォルダー名に使用できない文字が含まれています: {0}", x));
                    }
                    else if (x == "." || x == "..")
                    {
                        throw new ArgumentException("ソリューションフォルダー名に '.' または '..' という名前は使えません。");
                    }
                    else if (System.Text.RegularExpressions.Regex.IsMatch(x, @"^(?:CON|AUX|PRN|NUL|(?:COM|LPT)[0-9]+)(?:$|\.+.*$)"))
                    {
                        throw new ArgumentException("ソリューションフォルダー名に 'CON', 'AUX', 'PRN', 'COM1' または 'LPT2' のようなシステム予約名は使えません。");
                    }
                    else
                    {
                        return x;
                    }
                }));
        }

        private static IEnumerable<TSource> Flatten<TSource>(IEnumerable<TSource> source, Func<TSource, IEnumerable<TSource>> selector)
        {
            foreach (var x in source)
            {
                yield return x;
                foreach (var y in Flatten(selector(x), selector))
                {
                    yield return y;
                }
            }
        }

        #region UUID Version 5 の生成処理
        // 実装は一林さん謹製のものをベースに作成
        // ssh://git@spdlybra.nintendo.co.jp:7999/~ichibayashi_hironori/ichilib.git より取得可能 (2016.02.29)

        // UUID 仕様:
        // http://www.ietf.org/rfc/rfc4122.txt
        // http://www.rfc-editor.org/errata_search.php?rfc=4122&eid=1352

        /// <summary>
        /// RFC4122 4.3 の方法で UUID Version 5 を生成します。
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        private static Guid CreateGuidFromStringWithSHA1(string source)
        {
            using (var generator = System.Security.Cryptography.SHA1.Create())
            {
                // GUID のフィールドをネットワークバイトオーダーで表したバイト列 (UUID) を取得します。
                var uuid = SwapBetweenGuidAndUuid(s_NamespaceGuid.ToByteArray());
                // UUID 名前空間 + 入力文字列からハッシュ値を計算します。
                var hash = generator.ComputeHash(uuid.Concat(Encoding.UTF8.GetBytes(source)).ToArray());
                // Version
                var version = 5;
                hash[6] = (byte)((hash[6] & 0x0f) | ((version << 4) & 0xf0));
                // Variant = 10b
                hash[8] = (byte)((hash[8] & 0x3f) | 0x80);
                // UUID のバイトオーダーを反転して GUID 表現に直し、Guid クラスのインスタンスにします。
                return new Guid(SwapBetweenGuidAndUuid(hash).Take(16).ToArray());
            }
        }

        /// UUID と UUID 間のバイトオーダー反転を行います。
        private static byte[] SwapBetweenGuidAndUuid(byte[] byteArray)
        {
            // GUID と UUID では最初の 3 つのフィールドのバイトオーダーが逆であるため、反転します。
            Array.Reverse(byteArray, 0, 4);
            Array.Reverse(byteArray, 4, 2);
            Array.Reverse(byteArray, 6, 2);

            return byteArray;
        }
        #endregion

        public class SolutionProject
        {
            public string Path { get; set; }
            public IEnumerable<SolutionProject> Dependencies { get; set; }

            public string SolutionFolderPath { get; set; }
            public IEnumerable<SolutionProject> Projects { get; set; }

            public SolutionProject()
                : this(null, null)
            {
            }
            public SolutionProject(string path)
                : this(path, Enumerable.Empty<SolutionProject>())
            {
            }
            public SolutionProject(string path, IEnumerable<SolutionProject> dependencies)
            {
                Path = path;
                Dependencies = dependencies;
                Projects = Enumerable.Empty<SolutionProject>();
            }
        }
    }
}
