<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Install Visual Studio Help of NintendoSDK to Visual Studio 2017 environment.

    .DESCRIPTION
        Install Visual Studio Help (VsHelp) to Visual Studio 2017 environment.
        If Microsoft.Component.HelpViewer is not installed, this script also install it from internet.
#>

$scriptPath = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

$VsInstallerDirectoryPath = "C:\Program Files (x86)\Microsoft Visual Studio\Installer"
$VswherePath = Join-Path $VsInstallerDirectoryPath vswhere.exe
$VsInstalerShellPath = Join-Path $VsInstallerDirectoryPath vs_installershell.exe

$HelpViewerCtntMgrPath = "C:\Program Files (x86)\Microsoft Help Viewer\v2.3\HlpCtntMgr.exe"

function CheckVS2017Installed()
{
    Param([ref]$VsInstanceId, [ref]$VsChannelId)

    Write-Verbose "Check if VS2017 is installed"

    if(-not(Test-Path $VsInstallerDirectoryPath))
    {
        throw "Visual Studio after 2017 is NOT installed."
    }

    if(-not(Test-Path $VswherePath))
    {
        throw "vswhere.exe is NOT found. ($VswherePath)"
    }

    $VsWhereArguments = "-format value -property instanceId -version 15.0 -products Microsoft.VisualStudio.Product.Professional"

    $exitCode = -1
    $log = ""
    ExecuteProcess $VswherePath $VsWhereArguments ([ref]$exitCode) ([ref]$log)

    $instanceId = $log.Trim()

    if(($exitCode -eq 0) -and $instanceId)
    {
        $VsInstanceId.Value = $instanceId
    }
    else
    {
        throw "Visual Studio 2017 Professional is NOT installed."
    }

    $VsWhereArguments = "-format value -property channelId -version 15.0 -products Microsoft.VisualStudio.Product.Professional"
    ExecuteProcess $VswherePath $VsWhereArguments ([ref]$exitCode) ([ref]$log)
    $channelId = $log.Trim()

    if(($exitCode -eq 0) -and $channelId)
    {
        $VsChannelId.Value = $channelId
    }
    else
    {
        throw "The channelId of Visual Studio 2017 Professional is NOT installed."
    }
}

function CheckHelpViewerInstalled()
{
    Param([ref]$HelpViewerInstalled)

    Write-Verbose "Check if Help Viewer is installed"

    if(-not(Test-Path $HelpViewerCtntMgrPath))
    {
        $HelpViewerInstalled.Value = $false;
    }
    else
    {
        $HelpViewerInstalled.Value = $true;
    }
}

function InstallHelpViewer()
{
    Param($VsChannelId)

    Write-Host "Install Microsoft.Component.HelpViewer ."

    $exitCode = -1
    $log = ""
    $HelpViewerInstallArguments = "modify --productId Microsoft.VisualStudio.Product.Professional --channelId $VsChannelId --add Microsoft.Component.HelpViewer --noUpdateInstaller --quiet"
    # $HelpViewerInstallArguments = "modify --productId Microsoft.VisualStudio.Product.Professional --channelId $VsChannelId --add Microsoft.Component.HelpViewer --noUpdateInstaller --passive"

    ExecuteProcessWithAdmin $VsInstalerShellPath $HelpViewerInstallArguments ([ref]$exitCode) ([ref]$log)

    if($exitCode -ne 0)
    {
        throw "Failed to install HelpViewer of VS2017. (exitCode=$exitCode)`nPlease install Microsoft.Component.HelpViewer manually."
    }
}

function InstallSdkVsHelp()
{
    $sdkRootPath = GetSdkRootDirectory
    $VsHelpMshaPath = Join-Path $sdkRootPath "Documents\Api\VsHelp\helpcontentsetup.msha"
    $langPath = Join-Path $sdkRootPath "Documents\Api\VsHelp\lang.txt"

    Write-Host "Install SDK help file."

    if(-not(Test-Path $VsHelpMshaPath))
    {
        throw "The help file is not found. Please install NintendoSDK Document package. (path = $VsHelpMshaPath)"
    }

    $exitCode = -1
    $log = ""
    $lang = GetLangFromText $langPath
    $VshelpInstallArguments = "/operation install /catalogname VisualStudio15 /locale $lang /silent /sourceuri `"$VsHelpMshaPath`""
    ExecuteProcessWithAdmin $HelpViewerCtntMgrPath $VshelpInstallArguments ([ref]$exitCode) ([ref]$log)

    if($exitCode -ne 0)
    {
        throw "Failed to install help file of VS2017. (exitCode=$exitCode, path=$VsHelpMshaPath)"
    }
}

function GetLangFromText()
{
    Param($textPath)

    $defaultLang = "ja-JP"

    if(-not(Test-Path $textPath))
    {
        return $defaultLang
    }

    $text1stLine = Get-Content -Path $textPath -TotalCount 1
    $langText = $text1stLine.Trim()
    if($langText -eq "")
    {
        return $defaultLang
    }
    return $langText
}

function ExecuteProcess
{
    Param($execPath, $arguments, [ref]$exitCode, [ref]$log)

    if(-not(Test-Path $execPath))
    {
        throw "Execute file not found.($execPath)"
    }

    $processInfo = New-Object System.Diagnostics.ProcessStartInfo
    $processInfo.FileName = $execPath
    $processInfo.RedirectStandardError = $true
    $processInfo.RedirectStandardOutput = $true
    $processInfo.UseShellExecute = $false
    $processInfo.Arguments = $arguments

    $process = New-Object System.Diagnostics.Process
    $process.StartInfo = $processInfo

    $process.Start() | Out-Null
    $process.WaitForExit()

    $log.Value = $process.StandardOutput.ReadToEnd()
    # $errLog = $process.StandardError.ReadToEnd()
    $exitCode.Value = $process.exitCode
}

function ExecuteProcessWithAdmin
{
    Param($execPath, $arguments, [ref]$exitCode, [ref]$log)

    if(-not(Test-Path $execPath))
    {
        throw "Execute file not found.($execPath)"
    }

    $processInfo = New-Object System.Diagnostics.ProcessStartInfo
    $processInfo.FileName = $execPath
    $processInfo.RedirectStandardError = $false
    $processInfo.RedirectStandardOutput = $false
    $processInfo.UseShellExecute = $true
    $processInfo.Arguments = $arguments
    $processInfo.Verb = "runas"

    $process = New-Object System.Diagnostics.Process
    $process.StartInfo = $processInfo

    $process.Start() | Out-Null
    $process.WaitForExit()

    # $log.Value = $process.StandardOutput.ReadToEnd()
    # $errLog = $process.StandardError.ReadToEnd()
    $exitCode.Value = $process.exitCode
}

function GetSdkRootDirectory()
{
    $SdkRootMark = "NintendoSdkRootMark"

    $scanDirectoryPath = $scriptDirectoryPath
    while($scanDirectoryPath -ne "")
    {
        $checkRootPath = Join-Path $scanDirectoryPath $SdkRootMark
        if(Test-Path $checkRootPath)
        {
            return $scanDirectoryPath
        }
        $scanDirectoryPath = Split-Path -Parent $scanDirectoryPath
    }
    throw "Not found NintendoSDK root directory."
}


$VsInstanceId = $null
$VsChannelId = $null
$isInstalled = $false

Write-Verbose "InstallVsHelp.ps1 start"

# VS2017 がインストールされていることを確認する
CheckVS2017Installed ([ref]$VsInstanceId) ([ref]$VsChannelId)

# Microsoft.Component.HelpViewer がインストールされていることを確認する
CheckHelpViewerInstalled ([ref]$isInstalled)
if($isInstalled -eq $false)
{
    # Microsoft.Component.HelpViewer をインストールする
    InstallHelpViewer $VsChannelId
}

# NintendoSDK 用 VsHelp をインストールする
InstallSdkVsHelp

Write-Host "Install NintendoSDK Help file for Visual Studio 2017 Succeeded."

exit 0
