﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using GitExternalStorage.Transports;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalStorage.Git;

namespace GitExternalStorage.Configs
{
    /// <summary>
    /// Git の config から設定を読み出すクラスです。
    /// </summary>
    public class ConfigFromGit
    {
        private GitRepositoryBase gitRepository;

        public ConfigFromGit(GitRepositoryBase gitRepository)
        {
            this.gitRepository = gitRepository;
        }

        /// <summary>
        /// 設定値を読み出します。
        /// </summary>
        /// <returns></returns>
        public ConfigStruct ReadConfig()
        {
            var config = new ConfigStruct();

            config.Transport = ReadTransport();
            config.AutoDownload = ReadAutoDownload();

            switch (config.Transport)
            {
                case TransportType.Local:
                    config.Local = new LocalTransportStruct();
                    config.Local.Path = ReadLocalPath();
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }

            return config;
        }

        /// <summary>
        /// 設定値を書き込みます。
        /// </summary>
        /// <param name="config"></param>
        public void WriteConfig(ConfigStruct config)
        {
            switch (config.Transport)
            {
                case TransportType.Local:
                    WriteTransport(config.Transport);
                    WriteAutoDownload(config.AutoDownload);
                    WriteLocalPath(config.Local.Path);
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        private TransportType ReadTransport()
        {
            string transport;

            try
            {
                transport = this.gitRepository.GetConfig("GitExternalStorage.transport");
            }
            catch (GitRepositoryCommandFailedException e)
            {
                throw new ConfigGitFormatException("値が設定されていません。", e)
                {
                    Data = { { "Detail", "GitExternalStorage.transport の値が設定されていません。" } }
                };
            }

            try
            {
                return (TransportType)Enum.Parse(typeof(TransportType), transport, true);
            }
            catch (ArgumentException e)
            {
                throw new ConfigGitFormatException("設定値が不正です。", e)
                {
                    Data = { { "Detail", string.Format("取得した値 '{0}' は GitExternalStorage.transport の設定値として不正です。", transport) } }
                };
            }
        }
        private void WriteTransport(TransportType type)
        {
            gitRepository.SetConfig("GitExternalStorage.transport", type.ToString().ToLower());
        }

        private bool ReadAutoDownload()
        {
            string autodownload;
            try
            {
                autodownload = this.gitRepository.GetConfig("GitExternalStorage.autodownload");
            }
            catch (GitRepositoryCommandFailedException e)
            {
                throw new ConfigGitFormatException("値が設定されていません。", e)
                {
                    Data = { { "Detail", "GitExternalStorage.autodownload の値が設定されていません。" } }
                };
            }

            try
            {
                return bool.Parse(autodownload);
            }
            catch (FormatException e)
            {
                throw new ConfigGitFormatException("設定値が不正です。", e)
                {
                    Data = { { "Detail", string.Format("取得した値 '{0}' は GitExternalStorage.autodownload の設定値として不正です。", autodownload) } }
                };
            }
        }
        private void WriteAutoDownload(bool autodownload)
        {
            gitRepository.SetConfig("GitExternalStorage.autodownload", autodownload.ToString().ToLower());
        }

        private string ReadLocalPath()
        {
            try
            {
                var path = this.gitRepository.GetConfig("GitExternalStorage.localpath");
                if (path == string.Empty)
                {
                    throw new ConfigGitFormatException("値が設定されていません。")
                    {
                        Data = { { "Detail", "GitExternalStorage.localpath の値が設定されていません。" } }
                    };
                }

                return path;
            }
            catch (GitRepositoryCommandFailedException e)
            {
                throw new ConfigGitFormatException("値が設定されていません。", e)
                {
                    Data = { { "Detail", "GitExternalStorage.localpath の値が設定されていません。" } }
                };
            }
        }
        private void WriteLocalPath(string path)
        {
            gitRepository.SetConfig("GitExternalStorage.localpath", path);
        }
    }
}
