﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using GitExternalStorage.Consoles;
using GitExternalStorage.Core;
using GitExternalStorage.Transports;

namespace GitExternalStorage.Commands
{
    /// <summary>
    /// ハッシュ化ファイルを展開するコマンドです。必要に応じて、外部ストレージからファイルをダウンロードします。
    /// </summary>
    public class ExpandCommand : CommandBase
    {
        public ExpandCommand()
            : base()
        {
        }

        public ExpandCommand(CommandConsoleBase console)
            : base(console)
        {
        }

        public override void Run()
        {
            this.ExpandReferences();
        }

        /// <summary>
        /// ハッシュ化されたままのファイルを展開します。
        /// </summary>
        private void ExpandReferences()
        {
            CommandConsole.WriteLineToOut("Gathering files to be expanded");
            var referenceInfo = Reference.FindReferences(Environments.GitRepository);

            if (referenceInfo.ToExpand.Count() > 0)
            {
                // ダウンロードスレッドを作成
                var dummyCacheFiles = new List<string>();
                var failToExpandFiles = new List<ReferenceEntry>();
                Libgit2.ExpandSmudgeFilter.ResetNumberOfRunningCacheDownloadThreads(1);
                var cacheFileDownloadThread = new System.Threading.Thread(() =>
                    {
                        ExpandWithLibgit2DownloadCacheFileThreadFunction(
                            dummyCacheFiles,
                            failToExpandFiles,
                            referenceInfo.ToExpand);
                    });
                try
                {
                    // ダウンロードスレッドを開始
                    CommandConsole.WriteLineToOut("Start Downloading cache files");
                    cacheFileDownloadThread.Start();

                    // チェックアウトを実行
                    {
                        CommandConsole.WriteLineToOut("Preparing Expansion");
                        foreach (var entry in referenceInfo.ToExpand)
                        {
                            // 強制的にチェックアウトさせるために現在のファイルを削除
                            File.Delete(entry.FilePath);
                        }

                        CommandConsole.WriteLineToOut("Start Expansion");
                        Environments.GitRepositoryLibgit2.Expand(referenceInfo.ToExpand.Select(x => x.FilePath).ToList(), Environments.GetCacheDirectoryPath());
                    }
                }
                finally
                {
                    CommandConsole.WriteLineToOut("Finalizing Expansion");

                    // ダウンロードスレッドの終了を待つ
                    try
                    {
                        cacheFileDownloadThread.Join();
                    }
                    catch (Exception)
                    {
                    }

                    // 作成されたダミーのキャッシュファイルを削除
                    foreach (var path in dummyCacheFiles)
                    {
                        File.Delete(path);
                    }

                    // 展開に失敗したファイルがあるなら例外を投げて終了する
                    if(failToExpandFiles.Count > 0)
                    {
                        var stringBuilder = new StringBuilder();

                        stringBuilder.AppendFormat("Failed to expand the following files.\n");
                        foreach(var f in failToExpandFiles)
                        {
                            stringBuilder.AppendFormat("  {0} ({1})\n", f.FilePath, f.Sha);
                        }

                        throw new GitExternalStorageException(stringBuilder.ToString());
                    }
                }
            }

            CommandConsole.WriteLineToOut("Complete!");
        } // void ExpandReferences()

        private void ExpandWithLibgit2DownloadCacheFileThreadFunction(
            List<string> dummyCacheFiles,
            List<ReferenceEntry> failToExpandFiles,
            List<ReferenceEntry> toExpandFiles)
        {
            try
            {
                var cacheDirectory = Environments.GetCacheDirectoryPath();
                int totalExpand = toExpandFiles.Count;
                int countDownload = 0;
                foreach (var entry in toExpandFiles)
                {
                    countDownload++;
                    var cacheFilePath = Path.Combine(cacheDirectory, entry.Sha);
                    if (!File.Exists(cacheFilePath))
                    {
                        CommandConsole.WriteLineToOut("Downloading({0}/{1}) file: {2} ({3})", countDownload, totalExpand, entry.Sha, entry.FilePath);

                        var transport = Environments.Transport;
                        var cdnTransport = Environments.CdnTransport;
                        var tmpFilePath = Environments.GetTemporaryFilePath();

                        // CDN 設定があるならば、CDN からの取得を試みる
                        if (cdnTransport != null)
                        {
                            CommandConsole.WriteLineToError("Try to download file: {0}/{1}", cdnTransport.GetPath(), entry.Sha);

                            try
                            {
                                cdnTransport.PullUsingTempFile(entry.Sha, tmpFilePath);

                                // ダウンロードできたので、次のファイルへ
                                continue;
                            }
                            catch (Exception e)
                            {
                                CommandConsole.WriteLineToError("Info: failed to download file: {0}/{1}", cdnTransport.GetPath(), entry.Sha);
                                CommandConsole.WriteLineToError("Info: {0}", e.Message);
                            }
                        }

                        // CDN 設定がない、あるいは CDN からダウンロードできなかった場合、本体からのダウンロードを試みる
                        CommandConsole.WriteLineToError("Try to download file: {0}/{1}", transport.GetPath(), entry.Sha);

                        try
                        {
                            transport.PullUsingTempFile(entry.Sha, tmpFilePath);
                        }
                        catch (Exception e)
                        {
                            CommandConsole.WriteLineToError("Error: failed to download file: {0}/{1}", transport.GetPath(), entry.Sha);
                            CommandConsole.WriteLineToError("Error: {0}", e.Message);

                            // 展開に失敗したファイルを記録する
                            failToExpandFiles.Add(entry);

                            // ダウンロードに失敗したらダミーのファイルを生成する
                            // チェックアウトのスレッドではファイルが生成されたことを以って
                            // そのファイルのダウンロードの処理が完了したことを検出する。
                            // ハッシュ値が書き込まれたファイル（つまり展開前と同内容のファイル）を生成することで
                            // 展開の前後で内容が変わらないようにする。
                            try
                            {
                                using (var sw = new System.IO.StreamWriter(tmpFilePath))
                                {
                                    sw.Write(HashedContent.CreateFromHash(entry.Sha).Content);
                                    sw.Close();
                                }
                                File.Move(tmpFilePath, cacheFilePath);
                                dummyCacheFiles.Add(cacheFilePath);
                            }
                            catch
                            {
                                File.Delete(cacheFilePath);
                            }
                            finally
                            {
                                File.Delete(tmpFilePath);
                            }
                        }
                    }
                } // foreach (var entry in toExpandFiles)
            }
            catch (Exception)
            {
                // スレッド内で例外が発生したらスレッドを終了させる
            }
            finally
            {
                Libgit2.ExpandSmudgeFilter.DecrementNumberOfRunningCacheDownloadThreads();
            }
        } // void ExpandWithLibgit2DownloadCacheFileThreadFunction(...)
    }
}
