﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalRepositoryManager.Exceptions;
using GitExternalRepositoryManager.Exceptions.Git;
using GitExternalRepositoryManager.Git;

namespace GitExternalRepositoryManager.Commands
{
    /// <summary>
    /// 設定された属性値に対応する外部リポジトリを exrepo で初期化します。
    /// </summary>
    public class InitCommand
    {
        public class Parameters
        {
        }

        public void Execute(Parameters parameters)
        {
            Console.WriteLine("Initialize external repositories.");

            var repositoryRoot = GetRepositoryRoot();
            var attributes = GetAttributes();
            var directoryPath = GetDirectoryPath();

            // 利用登録すべき外部リポジトリのリスト
            var externalRepositories = new HashSet<string>();

            // 属性値に対応する外部リポジトリを一覧する
            foreach (var attribute in attributes)
            {
                var filename = attribute + ".txt";
                var filepath = Path.Combine(repositoryRoot, directoryPath, filename);

                Console.WriteLine("Read entries from '{0}'", filepath);

                try
                {
                    if (!File.Exists(filepath))
                    {
                        // リビジョン間を行き来することで、過去のリビジョンには存在しない属性が設定されている
                        // 場合があるため、属性に対応するリポジトリリストがなくても、エラーにはしない。
                        Console.WriteLine("Info: Attribute '{0}' does not have a repository list file.", attribute);
                        continue;
                    }

                    foreach (var entry in File.ReadAllLines(filepath))
                    {
                        externalRepositories.Add(entry);
                    }
                }
                catch (IOException e)
                {
                    throw new GitExternalRepositoryManagerException(string.Format("Failed to read '{0}'.", filepath), e);
                }
            }

            // 空行は取り除きます。
            externalRepositories.RemoveWhere(x => x == string.Empty);

            // 外部リポジトリを exrepo で初期化します。
            foreach (var externalRepository in externalRepositories)
            {
                Console.WriteLine("Initialize {0}", externalRepository);
                InitializeExternalRepository(externalRepository);
            }
        }

        // リポジトリのルートパス
        private string GetRepositoryRoot()
        {
            return GitCommandUtility.Run("rev-parse --show-toplevel");
        }

        // 属性値。カンマ区切りで複数の属性が指定される
        private IEnumerable<string> GetAttributes()
        {
            try
            {
                return GitCommandUtility.Run("config --get exrepo-m.attributes").Split(new[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
            }
            catch (GitRepositoryOperationFailedException e)
            {
                throw new GitExternalRepositoryManagerException("Failed to get configuration 'exrepo-m.attributes'.", e);
            }
        }

        // 定義ファイルの置き場所
        private string GetDirectoryPath()
        {
            try
            {
                return GitCommandUtility.Run("config --get exrepo-m.directory");
            }
            catch (GitRepositoryOperationFailedException e)
            {
                throw new GitExternalRepositoryManagerException("Failed to get configuration 'exrepo-m.directory'.", e);
            }
        }

        private void InitializeExternalRepository(string externalRepository)
        {
            try
            {
                GitCommandUtility.Run(string.Format("exrepo init {0}", externalRepository));
            }
            catch (GitRepositoryOperationFailedException e)
            {
                throw new GitExternalRepositoryManagerException(string.Format("Failed to initialize external repository '{0}'.", externalRepository), e);
            }
        }
    }
}
